import { ClockCircleOutlined, CloudServerOutlined, CloudUploadOutlined } from "@ant-design/icons";
import { App, List, Space, Spin, Tooltip } from "antd";
import { observer } from "mobx-react";

import { copyToClipboard } from "@devowl-wp/react-utils";

import { PluginUpdateLicenseTelemetryDataModal } from "./licenseTelemetryDataModal.js";
import { __, _i } from "../../utils/i18n.js";

import type { License } from "../../models/license.js";
import type { FC } from "react";

const PluginUpdateLicenseListItem: FC<{ license: License; onDeactivate?: (license: License) => void }> = observer(
    ({ license, onDeactivate }) => {
        const { message } = App.useApp();
        const {
            busy,
            uuid,
            installationType,
            telemetryDataSharingOptIn,
            blogName,
            code,
            remote,
            programmatically,
            host,
            blog,
            store: { needsLicenseKeys },
        } = license;

        return (
            <Spin spinning={busy}>
                <List.Item
                    style={{ paddingLeft: 0, paddingRight: 0 }}
                    actions={[
                        needsLicenseKeys && (
                            <Space key="installationType">
                                <Tooltip title={__("Client UUID: %s, click to copy.", uuid)}>
                                    <CloudServerOutlined
                                        style={{ cursor: "pointer" }}
                                        onClick={() => {
                                            copyToClipboard(uuid);
                                            message.success("Successfully copied client UUID key to clipboard!");
                                        }}
                                    />
                                </Tooltip>
                                {installationType === "production"
                                    ? __("Production")
                                    : installationType === "development"
                                      ? __("Development")
                                      : "n/a"}
                            </Space>
                        ),
                        remote && (
                            <Space key="activatedAt">
                                <ClockCircleOutlined />
                                {__(
                                    "Activated %s",
                                    new Date(remote.licenseActivation.activatedAt).toLocaleString(
                                        document.documentElement.lang,
                                    ),
                                )}
                            </Space>
                        ),
                        <PluginUpdateLicenseTelemetryDataModal
                            key="telemetryModal"
                            license={license}
                            disabled={!telemetryDataSharingOptIn}
                        >
                            <Space key="telemetryDataSharingOptIn" style={{ cursor: "pointer" }}>
                                <CloudUploadOutlined />
                                {telemetryDataSharingOptIn
                                    ? __("Telemetry data sharing enabled")
                                    : __("Telemetry data sharing disabled")}
                            </Space>
                        </PluginUpdateLicenseTelemetryDataModal>,
                        !programmatically && (
                            <a
                                key="deactivate"
                                className="button-link"
                                onClick={async () => {
                                    try {
                                        await license.deactivate();
                                    } catch (e) {
                                        message.error((e as any).responseJSON.message);
                                    }
                                    onDeactivate?.(license);
                                }}
                            >
                                {__("Deactivate")}
                            </a>
                        ),
                    ].filter(Boolean)}
                >
                    <List.Item.Meta
                        title={blogName}
                        description={
                            needsLicenseKeys ? (
                                <>
                                    {__("Your license key")}: <code>{code}</code>
                                </>
                            ) : (
                                <>
                                    {__("Your installation is fully activated.")}
                                    <div style={{ marginTop: 15 }}>
                                        <a
                                            rel="noreferrer"
                                            href={__("https://devowl.io/knowledge-base/upgrade-free-to-pro-version/")}
                                            className="button"
                                            target="_blank"
                                        >
                                            {__("Enter license key to unlock PRO features")}
                                        </a>
                                    </div>
                                </>
                            )
                        }
                    />
                    {programmatically && (
                        <div className="notice notice-info inline below-h2 notice-alt" style={{ margin: "0 0 10px 0" }}>
                            <p>
                                {_i(
                                    __(
                                        "This license cannot be deactivated manually because it is configured programmatically. That means you have used the {{a}}activation filter{{/a}} for host {{code}}%s{{/code}} (Blog ID: %d). Please remove the filter to deactivate the license!",
                                        host,
                                        blog,
                                    ),
                                    {
                                        code: <code />,
                                        a: (
                                            <a
                                                href="https://docs.devowl.io/real-cookie-banner/hooks/DevOwl_RealProductManager_License_Programmatic_$slug.html"
                                                target="_blank"
                                                rel="noreferrer"
                                            />
                                        ),
                                    },
                                )}
                            </p>
                        </div>
                    )}
                </List.Item>
            </Spin>
        );
    },
);

export { PluginUpdateLicenseListItem };
