import { Divider, Modal, Spin } from "antd";
import { observer } from "mobx-react";
import { useCallback, useEffect, useState } from "react";

import { PluginUpdateErrorNotice } from "./embed.js";
import { PluginUpdateForm } from "./form/form.js";
import { PluginUpdateLicenseList } from "./licenseList.js";
import { useStores } from "../../store/stores.js";
import { __ } from "../../utils/i18n.js";

import type { FC } from "react";

const PluginUpdateModal: FC = observer(() => {
    const [doReloadAfterClose, setDoReloadAfterClose] = useState(false);
    const { pluginUpdateStore } = useStores();
    const { busy, modalPlugin, pluginUpdates, pluginUpdateFetchErrors } = pluginUpdateStore;
    const pluginUpdate = modalPlugin ? pluginUpdates.get(modalPlugin) : undefined;
    const showHeadline = pluginUpdate?.unlicensedEntries.length > 0 && pluginUpdate?.licensedEntries.length > 0;

    const handleCancel = useCallback(() => {
        if (doReloadAfterClose) {
            const { checkUpdateLink } = pluginUpdate;
            if (checkUpdateLink) {
                window.location.href = checkUpdateLink;
            } else {
                window.location.reload();
            }
        } else {
            pluginUpdateStore.hideModal();
        }
    }, [doReloadAfterClose, pluginUpdateStore, pluginUpdate]);

    const handleChange = useCallback(() => setDoReloadAfterClose(true), [setDoReloadAfterClose]);
    const error = pluginUpdateFetchErrors.get(modalPlugin);

    useEffect(() => {
        document.body.classList[modalPlugin ? "add" : "remove"]("rpm-wpc-antd-modal-open");
    }, [modalPlugin]);

    return modalPlugin ? (
        <Modal
            open={true}
            okButtonProps={{
                form: `license-form-${pluginUpdate?.slug}`,
                htmlType: "submit",
                style: { display: pluginUpdate?.unlicensedEntries.length === 0 || error ? "none" : undefined },
            }}
            cancelButtonProps={{ style: { display: "none" } }}
            onCancel={handleCancel}
            okText={__("Save")}
            title={
                pluginUpdate ? (
                    <span style={{ fontWeight: "normal" }}>
                        <strong>{pluginUpdate.name}:</strong> {__("License settings")}
                    </span>
                ) : (
                    ""
                )
            }
            width={800}
        >
            {error ? (
                <PluginUpdateErrorNotice slug={modalPlugin} />
            ) : busy || !pluginUpdate ? (
                <Spin spinning />
            ) : (
                <div>
                    {showHeadline && (
                        <Divider type="horizontal" orientation="left" style={{ marginTop: 0 }}>
                            {__("Not yet licensed")}
                        </Divider>
                    )}
                    <PluginUpdateForm onSave={handleChange} pluginUpdate={pluginUpdate} />
                    {showHeadline && (
                        <Divider type="horizontal" orientation="left">
                            {__("Already licensed")}
                        </Divider>
                    )}
                    {pluginUpdate.licensedEntries.length > 0 && (
                        <PluginUpdateLicenseList onDeactivate={handleChange} pluginUpdate={pluginUpdate} />
                    )}
                </div>
            )}
        </Modal>
    ) : null;
});

export { PluginUpdateModal };
