import { App, Switch, Tooltip } from "antd";
import { observer } from "mobx-react";
import { useCallback } from "react";

import { useStores } from "../../store/stores.js";
import { __ } from "../../utils/i18n.js";
import { ProTag } from "../proTag.js";

import type { FC } from "react";

const OptionsPostTypesRow: FC<{ name: string }> = observer(({ name }) => {
    const { message, modal } = App.useApp();
    const { optionStore } = useStores();
    const {
        others: { postTypes, isPro, pluginCptUi },
    } = optionStore;

    const { label, link, available, active, fastMode } = postTypes[name];

    const handleUpdatePostTypeActive = useCallback(
        async (state: boolean) => {
            await optionStore.updatePostTypeOptions(name, { active: state });
            message.success(
                state ? __("Category tree enabled for %s.", label) : __("Category tree disabled for %s.", label),
            );
        },
        [optionStore, label],
    );

    const handleUpdatePostTypeFastMode = useCallback(
        async (state: boolean) => {
            await optionStore.updatePostTypeOptions(name, { fastMode: state });
            message.success(
                state
                    ? __("Pagination without reloading the page enabled for %s.", label)
                    : __("Pagination without reloading the page disabled for %s.", label),
            );
        },
        [optionStore, label],
    );

    const handleCreateCustomPostType = useCallback(async () => {
        if (!pluginCptUi.active) {
            modal.confirm({
                cancelText: __("Cancel"),
                okText: "Continue",
                title: __("Custom Post Type UI"),
                content: __(
                    "To create custom taxonomies we highly recommend to use the 3rd party plugin Custom Post Type UI. Would you like to install and activate it now?",
                ),
                onOk: async () => {
                    await optionStore.installAndActivateCustomPostTypeUI();
                    window.location.href = pluginCptUi.manageTaxonomiesUrl;
                },
            });
        } else {
            window.location.href = pluginCptUi.manageTaxonomiesUrl;
        }
    }, [optionStore, pluginCptUi]);

    return (
        <tr>
            <td>
                <strong>{label}</strong> &bull;{" "}
                <a href={link} target="_blank" rel="noreferrer">
                    {__("Open")}
                </a>
            </td>
            <td>
                {name !== "post" && !isPro ? (
                    <ProTag feature="options-active" />
                ) : available ? (
                    <Switch checked={active} onChange={handleUpdatePostTypeActive} />
                ) : (
                    <Tooltip
                        transitionName={null}
                        placement="top"
                        title={__(
                            "There are no taxonomies available for this post type. You need to register your own taxonomy so that you can organize your content into its categories.",
                        )}
                    >
                        <button className="button" onClick={handleCreateCustomPostType}>
                            {__("Create custom taxonomy")}
                        </button>
                    </Tooltip>
                )}
            </td>
            <td>
                <Switch
                    checked={active && fastMode}
                    disabled={!active || !isPro}
                    onChange={handleUpdatePostTypeFastMode}
                />
            </td>
        </tr>
    );
});

export { OptionsPostTypesRow };
