<?php

namespace DevOwl\RealCookieBanner;

use DevOwl\RealCookieBanner\base\UtilsProvider;
use DevOwl\RealCookieBanner\settings\Consent;
use DevOwl\RealCookieBanner\settings\CountryBypass;
use DevOwl\RealCookieBanner\settings\General;
use DevOwl\RealCookieBanner\settings\GoogleConsentMode;
use DevOwl\RealCookieBanner\settings\Multisite;
use DevOwl\RealCookieBanner\settings\Revision;
use DevOwl\RealCookieBanner\settings\TCF;
use DevOwl\RealCookieBanner\view\customize\banner\FooterDesign;
use DevOwl\RealProductManagerWpClient\AbstractInitiator;
use DevOwl\RealProductManagerWpClient\license\TelemetryData;
use DevOwl\ServiceCloudConsumer\templates\ServiceTemplate;

// @codeCoverageIgnoreStart
defined('ABSPATH') or die('No script kiddies please!'); // Avoid direct file request
// @codeCoverageIgnoreEnd

/**
 * Initiate real-product-manager-wp-client functionality.
 */
class RpmInitiator extends AbstractInitiator {
    use UtilsProvider;

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function getPluginBase() {
        return $this;
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function getProductAndVariant() {
        return [1, $this->isPro() ? 1 : 2];
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function getPluginAssets() {
        return $this->getCore()->getAssets();
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function getPrivacyPolicy() {
        return 'https://devowl.io/privacy-policy';
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function isExternalUpdateEnabled() {
        return $this->isPro();
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function isAdminNoticeLicenseVisible() {
        return Core::getInstance()->getConfigPage()->isVisible();
    }

    /**
     * Documented in AbstractInitiator.
     *
     * @codeCoverageIgnore
     */
    public function isLocalAnnouncementVisible() {
        return $this->isAdminNoticeLicenseVisible();
    }

    /**
     * Allows you to build telemetry data.
     *
     * @param TelemetryData $telemetry
     */
    public function buildTelemetryData($telemetry) {
        $generalSettings = General::getInstance();
        $consentSettings = Consent::getInstance();
        $countryBypassSettings = CountryBypass::getInstance();
        $tcfSettings = TCF::getInstance();
        $gcmSettings = GoogleConsentMode::getInstance();
        $multisiteSettings = Multisite::getInstance();
        $stats = Stats::getInstance();
        $revision = Revision::getInstance();
        $revisionCurrent = $revision->getCurrent();
        $revisionIndependent = $revision->getRevision()->createIndependent();
        $decision = $revisionIndependent['revision']['banner']['customizeValuesBanner']['decision'];
        $blocker = $revisionIndependent['revision']['blocker'];
        $groups = $revisionCurrent['revision']['groups'];

        $counts = [
            'rcb_service_essential_count' => 0,
            'rcb_service_non-essential_legal-basis_consent_count' => 0,
            'rcb_service_non-essential_legal-basis_legitimate-interest_count' => 0,
            'rcb_service_non-essential_legal-basis_consent_with-visual-content-blocker_count' => 0,
            'rcb_service_non-essential_legal-basis_legitimate-interest_with-visual-content-blocker_count' => 0,
        ];

        // Service and content blockers
        $telemetry
            ->add('rcb_serviceGroup_count', count($groups))
            ->add('rcb_service_count', $revisionCurrent['all_cookie_count'])
            ->add('rcb_contentBlocker_count', $revisionCurrent['all_blocker_count'])
            ->add(
                'rcb_services',
                Utils::array_flatten(
                    array_map(function ($group) use (&$counts, $blocker) {
                        $items = [];
                        foreach ($group['items'] as $item) {
                            $newItem = ['name' => $item['name']];
                            if (!empty($item['presetId'])) {
                                $newItem['identifier'] = $item['presetId'];
                            }
                            $items[] = $newItem;

                            if ($group['isEssential']) {
                                $counts['rcb_service_essential_count']++;
                            } else {
                                // Check if there is a visual content blocker connected to this service
                                $visualContentBlocker = array_filter($blocker, function ($blockerItem) use ($item) {
                                    return in_array($item['id'], $blockerItem['services'], true);
                                });

                                if ($item['legalBasis'] === ServiceTemplate::LEGAL_BASIS_CONSENT) {
                                    $counts['rcb_service_non-essential_legal-basis_consent_count']++;

                                    if (!empty($visualContentBlocker)) {
                                        $counts[
                                            'rcb_service_non-essential_legal-basis_consent_with-visual-content-blocker_count'
                                        ]++;
                                    }
                                } elseif ($item['legalBasis'] === ServiceTemplate::LEGAL_BASIS_LEGITIMATE_INTEREST) {
                                    $counts['rcb_service_non-essential_legal-basis_legitimate-interest_count']++;

                                    if (!empty($visualContentBlocker)) {
                                        $counts[
                                            'rcb_service_non-essential_legal-basis_legitimate-interest_with-visual-content-blocker_count'
                                        ]++;
                                    }
                                }
                            }
                        }
                        return $items;
                    }, $groups)
                )
            )
            ->add(
                'rcb_contentBlockers',
                array_map(function ($blocker) {
                    $item = ['name' => $blocker['name']];
                    if (!empty($blocker['presetId'])) {
                        $item['identifier'] = $blocker['presetId'];
                    }
                    return $item;
                }, $blocker)
            )
            ->add(
                'rcb_scanner_externalUrls',
                array_values(
                    array_map(function ($item) {
                        return [
                            'host' => $item['host'],
                            'tags' => $item['tags'],
                            'count' => $item['foundOnSitesCount'],
                        ];
                    }, Core::getInstance()->getScanner()->getQuery()->getScannedExternalUrls())
                )
            );

        foreach ($counts as $key => $value) {
            $telemetry->add($key, $value);
        }

        // Stats
        $telemetry->add(
            'rcb_consent_count',
            UserConsent::getInstance()->byCriteria([], UserConsent::BY_CRITERIA_RESULT_TYPE_COUNT)
        );

        if ($this->isPro()) {
            $statsTimeFrom = gmdate('Y-m-d', strtotime('-30 days'));
            $statsTimeTo = gmdate('Y-m-d');

            $clickedButton = $stats->fetchButtonsClickedStats($statsTimeFrom, $statsTimeTo);
            $customBypass = $stats->fetchCustomBypassStats($statsTimeFrom, $statsTimeTo);
            $clickedButtonResult = [];
            $customBypassResult = [];
            foreach ($clickedButton as $key => $value) {
                $clickedButtonResult[] = [
                    'type' => $key,
                    'label' => $value[0],
                    'count' => $value[1],
                ];
            }
            foreach ($customBypass as $key => $value) {
                $customBypassResult[] = [
                    'type' => $key,
                    'count' => $value,
                ];
            }

            $telemetry->add('rcb_statistic_consentByClickedButton', $clickedButtonResult);
            $telemetry->add('rcb_statistic_consentByBypass', $customBypassResult);
        }

        $telemetry
            // Look & feel
            ->add('rcb_customizer_consentOptions_acceptAll', $decision['acceptAll'])
            ->add('rcb_customizer_consentOptions_acceptEssentials', $decision['acceptEssentials'])
            ->add('rcb_customizer_consentOptions_acceptIndividual', $decision['acceptIndividual'])
            // Settings
            ->add('rcb_settings_setCookiesViaManager', $generalSettings->getSetCookiesViaManager())
            ->add('rcb_settings_acceptAllForBots', $consentSettings->isAcceptAllForBots())
            ->add('rcb_settings_respectDoNotTrack', $consentSettings->isRespectDoNotTrack())
            ->add('rcb_settings_ageNotice', $consentSettings->isAgeNoticeEnabled())
            ->add('rcb_settings_listServicesNotice', $consentSettings->isListServicesNoticeEnabled())
            ->add('rcb_settings_countryBypass', $countryBypassSettings->isActive())
            ->add('rcb_settings_tcf', $tcfSettings->isActive())
            ->add('rcb_settings_gcm', $gcmSettings->isEnabled())
            ->add('rcb_settings_bannerLessConsent', $consentSettings->isBannerLessConsent())
            ->add('rcb_settings_dataProcessingInUnsafeCountries', $consentSettings->isDataProcessingInUnsafeCountries())
            ->add('rcb_settings_consentForwarding', $multisiteSettings->isConsentForwarding())
            // General
            ->add('rcb_wpPlugins_active', $telemetry->getActivePlugins())
            ->add('rcb_wpThemes_active', $telemetry->getActiveTheme());
    }

    // Documented in AbstractInitiator
    public function formAdditionalCheckboxes() {
        if ($this->isPro() || boolval(get_option(FooterDesign::SETTING_POWERED_BY_LINK))) {
            return [];
        }

        return [
            [
                'id' => 'powered-by-link',
                'text' => __(
                    'I allow to place a link in the cookie banner, which informs about the use of this plugin.',
                    RCB_TD
                ),
                'stateFn' => function ($state) {
                    if ($state) {
                        update_option(FooterDesign::SETTING_POWERED_BY_LINK, true);
                    }
                },
            ],
        ];
    }
}
