<?php
namespace DevOwl\RealCookieBanner\view\customize\banner;

use DevOwl\CookieConsentManagement\settings\AbstractGeneral;
use DevOwl\Customize\AbstractCustomizePanel;
use DevOwl\Customize\controls\TinyMCE;
use DevOwl\RealCookieBanner\base\UtilsProvider;
use DevOwl\RealCookieBanner\comp\language\Hooks;
use DevOwl\RealCookieBanner\Core;
use DevOwl\RealCookieBanner\settings\CookieGroup;
use DevOwl\RealCookieBanner\settings\General;
use DevOwl\RealCookieBanner\view\BannerCustomize;

// @codeCoverageIgnoreStart
defined('ABSPATH') or die('No script kiddies please!'); // Avoid direct file request
// @codeCoverageIgnoreEnd

/**
 * Cookie policy texts.
 */
class CookiePolicy {
    use UtilsProvider;

    const SECTION = BannerCustomize::PANEL_MAIN . '-cookie-policy';

    const SETTING = RCB_OPT_PREFIX . '-cookie-policy';

    const SETTING_INSTRUCTION = self::SETTING . '-instruction';
    const SETTING_HEADLINE_TABLE_OF_CONTENTS = self::SETTING . '-h-toc';
    const SETTING_HEADLINE_CONTROLLER_OF_WEBSITE = self::SETTING . '-h-website-operator';
    const SETTING_HEADLINE_DIFF_TO_PRIVACY_POLICY = self::SETTING . '-h-diff-privacy-policy';
    const SETTING_HEADLINE_COOKIE_TECHNOLOGY = self::SETTING . '-h-cookie-technology';
    const SETTING_HEADLINE_LEGAL_BASIS = self::SETTING . '-h-legal-basis';
    const SETTING_HEADLINE_RIGHTS_OF_THE_VISITOR = self::SETTING . '-h-rights-of-visitor';
    const SETTING_HEADLINE_MANAGE_COOKIES = self::SETTING . '-h-manage-cookies';
    const SETTING_HEADLINE_TYPES_OF_COOKIES = self::SETTING . '-h-types-cookies';
    const SETTING_HEADLINE_COOKIE_ORIGIN = self::SETTING . '-h-cookie-origin';
    const SETTING_HEADLINE_LIST_OF_SERVICES = self::SETTING . '-h-list-of-services';

    const SETTING_DIFF_TO_PRIVACY_POLICY = self::SETTING . '-diff-privacy-policy';
    const SETTING_COOKIE_TECHNOLOGY = self::SETTING . '-cookie-technology';
    const SETTING_LEGAL_BASIS_GDPR = self::SETTING . '-legal-basis-gdpr';
    const SETTING_LEGAL_BASIS_DSG = self::SETTING . '-legal-basis-dsg';
    const SETTING_RIGHTS_OF_THE_VISITOR = self::SETTING . '-rights-of-visitor';
    const SETTING_MANAGE_COOKIES = self::SETTING . '-manage-cookies';
    const SETTING_TYPES_OF_COOKIES = self::SETTING . '-types-cookies';
    const SETTING_SHOW_CUSTOM_GROUPS = self::SETTING . '-show-custom-groups';
    const SETTING_COOKIE_ORIGIN = self::SETTING . '-cookie-origin';
    const SETTING_IS_TABLE_DARK_MODE = self::SETTING . '-table-dark-mode';
    const SETTING_ADDITIONAL_CONTENT = self::SETTING . '-additional-content';

    const DEFAULT_SHOW_CUSTOM_GROUPS = true;
    const DEFAULT_IS_TABLE_DARK_MODE = false;

    /**
     * Return arguments for this section.
     */
    public function args() {
        $defaultTexts = self::getDefaultTexts();
        $generalSettings = General::getInstance();
        $address = $generalSettings->getOperatorContactAddress();
        $country = $generalSettings->getOperatorCountry();
        $email = $generalSettings->getOperatorContactEmail();
        $phone = $generalSettings->getOperatorContactPhone();
        $contactUrl = $generalSettings->getOperatorContactFormUrl();
        $territorialLegalBasis = $generalSettings->getTerritorialLegalBasis();
        $isTerritorialLegalBasisGdpr = in_array(
            AbstractGeneral::TERRITORIAL_LEGAL_BASIS_GDPR,
            $territorialLegalBasis,
            true
        );
        $isTerritorialLegalBasisDsg = in_array(
            AbstractGeneral::TERRITORIAL_LEGAL_BASIS_DSG_SWITZERLAND,
            $territorialLegalBasis,
            true
        );

        $nonDefaultGroups = [];
        // At this time, the taxonomy could not yet be registered (e.g. when updating the plugin)
        // As this is only for view purposes, we simply check if the taxonomy exists
        if (taxonomy_exists(CookieGroup::TAXONOMY_NAME)) {
            $nonDefaultGroups = Core::getInstance()
                ->getCookieConsentManagement()
                ->getSettings()
                ->getGeneral()
                ->getNonDefaultServiceGroups();
        }

        return [
            'name' => 'cookiePolicy',
            'title' => __('Cookie policy', RCB_TD),
            'description' => sprintf(
                '<p>%s</p>',
                join('</p><p>', [
                    __(
                        'A cookie policy is a document that lists all the cookies used on a website and provides comprehensive information about each cookie. They also help website visitors understand how their data is used, how long cookies remain on their device and other important information. Creating a cookie policy can help to clearly divide up the information on your website and avoid having to enter everything in the privacy policy. While your privacy policy encompasses details about all the data you gather, handle, store, or transmit, a cookie policy specifically focuses on cookies. There is <strong>no obligation</strong> to create a cookie policy under the GDPR or the ePrivacy Directive but shows your efforts to treat your website visitors fairly and transparently.',
                        RCB_TD
                    ),
                    __(
                        'Use <code>{{privacyPolicy}}privacy policy{{/privacyPolicy}}</code> as a placeholder for the privacy policy link.',
                        RCB_TD
                    ),
                ])
            ),
            'active_callback' => function () {
                return General::getInstance()->getCookiePolicyId() > 0;
            },
            'customParams' => [
                'url' => General::getInstance()->getCookiePolicyUrl(),
            ],
            'controls' => [
                self::SETTING_INSTRUCTION => [
                    'name' => 'instruction',
                    'label' => __('Instruction', RCB_TD),
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['instruction'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_TABLE_OF_CONTENTS => [
                    'name' => 'headlineToc',
                    'label' => __('Table of contents', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineToc'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_CONTROLLER_OF_WEBSITE => [
                    'name' => 'headlineControllerOfWebsite',
                    'label' => __('Controller of the website', RCB_TD),
                    'description' =>
                        sprintf(
                            // translators:
                            __(
                                'Controller is the website operator specified in %1$sSettings > General > Website operator details%2$s.',
                                RCB_TD
                            ),
                            '<a href="' .
                                esc_attr(Core::getInstance()->getConfigPage()->getUrl()) .
                                '#/settings/" target="_blank">',
                            '</a>'
                        ) .
                        (empty($address) || empty($country) || empty(join('', [$email, $phone, $contactUrl]))
                            ? $this->getWebsiteOperatorIncompleteNotice()
                            : ''),
                    'setting' => [
                        'default' => $defaultTexts['headlineControllerOfWebsite'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_DIFF_TO_PRIVACY_POLICY => [
                    'name' => 'headlineDiffToPrivacyPolicy',
                    'label' => __('Addition to the privacy policy', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineDiffToPrivacyPolicy'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_DIFF_TO_PRIVACY_POLICY => [
                    'name' => 'diffToPrivacyPolicy',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['diffToPrivacyPolicy'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_COOKIE_TECHNOLOGY => [
                    'name' => 'headlineCookieTechnology',
                    'label' => __('What are cookies and similar technologies?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineCookieTechnology'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_COOKIE_TECHNOLOGY => [
                    'name' => 'cookieTechnology',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['cookieTechnology'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_LEGAL_BASIS => [
                    'name' => 'headlineLegalBasis',
                    'label' => __('What is the legal basis for setting/reading cookies?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineLegalBasis'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_LEGAL_BASIS_GDPR => [
                    'enabled' => $isTerritorialLegalBasisGdpr,
                    'name' => 'legalBasisGdpr',
                    'description' => __(
                        'Use <code>{{euLegalBasis}}</code> as a placeholder to display the legal bases for the most important countries in which the website language is spoken.',
                        RCB_TD
                    ),
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['legalBasisGdpr'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_LEGAL_BASIS_DSG => [
                    'enabled' => $isTerritorialLegalBasisDsg,
                    'name' => 'legalBasisDsg',
                    'description' => $isTerritorialLegalBasisGdpr
                        ? __('Addition for the DSG in Switzerland', RCB_TD)
                        : '',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['legalBasisDsg'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_RIGHTS_OF_THE_VISITOR => [
                    'name' => 'headlineRightsOfTheVisitor',
                    'label' => __('What rights does the website visitor have?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineRightsOfTheVisitor'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_RIGHTS_OF_THE_VISITOR => [
                    'name' => 'rightsOfTheVisitor',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['rightsOfTheVisitor'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_MANAGE_COOKIES => [
                    'name' => 'headlineManageCookies',
                    'label' => __('How to manage cookies in a browser?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineManageCookies'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_MANAGE_COOKIES => [
                    'name' => 'manageCookies',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['manageCookies'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_TYPES_OF_COOKIES => [
                    'name' => 'headlineTypesOfCookies',
                    'label' => __('What types of cookies exist?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineTypesOfCookies'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_TYPES_OF_COOKIES => [
                    'name' => 'typesOfCookies',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['typesOfCookies'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_SHOW_CUSTOM_GROUPS => [
                    'name' => 'isShowCustomGroups',
                    'enabled' => count($nonDefaultGroups) > 0,
                    'label' => __('Mention of self-created service groups in the cookie policy', RCB_TD),
                    'type' => 'checkbox',
                    'setting' => [
                        'default' => self::DEFAULT_SHOW_CUSTOM_GROUPS,
                        'sanitize_callback' => [AbstractCustomizePanel::class, 'sanitize_checkbox'],
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_COOKIE_ORIGIN => [
                    'name' => 'headlineCookieOrigin',
                    'label' => __('Who can set cookies on this website?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineCookieOrigin'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_COOKIE_ORIGIN => [
                    'name' => 'cookieOrigin',
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'setting' => [
                        'default' => $defaultTexts['cookieOrigin'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_HEADLINE_LIST_OF_SERVICES => [
                    'name' => 'headlineListOfServices',
                    'label' => __('Which cookies are used on this website?', RCB_TD),
                    'setting' => [
                        'default' => $defaultTexts['headlineListOfServices'],
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_IS_TABLE_DARK_MODE => [
                    'name' => 'isTableDarkMode',
                    'label' => __('Show table in dark mode', RCB_TD),
                    'type' => 'checkbox',
                    'setting' => [
                        'default' => self::DEFAULT_IS_TABLE_DARK_MODE,
                        'sanitize_callback' => [AbstractCustomizePanel::class, 'sanitize_checkbox'],
                        'transport' => 'refresh',
                    ],
                ],
                self::SETTING_ADDITIONAL_CONTENT => [
                    'name' => 'additionalContent',
                    'label' => __('Further information / free text field', RCB_TD),
                    'description' => __(
                        'Use <code>{{dateOfUpdate}}</code> as a placeholder to show date of the cookie policy when it was last changed.',
                        RCB_TD
                    ),
                    'type' => 'textarea',
                    'class' => TinyMCE::class,
                    'mediaButtons' => true,
                    'toolbar1' => 'formatselect,bold,italic,bullist,numlist,link',
                    'setting' => [
                        'default' => $defaultTexts['additionalContent'],
                        'sanitize_callback' => 'wp_kses_post',
                        'localize_callback' => 'wp_kses_post',
                        'allowEmpty' => true,
                        'transport' => 'refresh',
                    ],
                ],
            ],
        ];
    }

    /**
     * Return a notice HTML for the customize description when RCB's website operator details are incomplete.
     */
    public static function getWebsiteOperatorIncompleteNotice() {
        return sprintf(
            '<div class="notice notice-warning inline below-h2 notice-alt" style="margin: 10px 0px 0px;"><p>%s</p></div>',
            sprintf(
                // translators:
                __(
                    'This section is currently not displayed in your cookie policy because the data provided is incomplete. Please set the website operator details at %1$sSettings > General > Website operator details%2$s!',
                    RCB_TD
                ),
                '<a href="' .
                    esc_attr(Core::getInstance()->getConfigPage()->getUrl()) .
                    '#/settings/" target="_blank">',
                '</a>'
            )
        );
    }

    /**
     * Get the default texts.
     */
    public static function getDefaultTexts() {
        $tempTd = Hooks::getInstance()->createTemporaryTextDomain();

        $defaults = [
            'instruction' => _x(
                'With this cookie policy, we would like to explain what cookies and cookie-like technologies are, how we and other service provider use them on this website, what information they collect, for how long and for what purposes we use them.',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'headlineToc' => _x('Table of contents', 'legal-text', Hooks::TD_FORCED),
            'headlineControllerOfWebsite' => _x('Controller of the website', 'legal-text', Hooks::TD_FORCED),
            'headlineDiffToPrivacyPolicy' => _x('Addition to the privacy policy', 'legal-text', Hooks::TD_FORCED),
            'headlineCookieTechnology' => _x(
                'What are cookies and similar technologies?',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'headlineLegalBasis' => _x(
                'What is the legal basis for setting/reading cookies?',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'headlineRightsOfTheVisitor' => _x(
                'What rights does the website visitor have?',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'headlineManageCookies' => _x('How to manage cookies in a browser?', 'legal-text', Hooks::TD_FORCED),
            'headlineTypesOfCookies' => _x('What types of cookies exist?', 'legal-text', Hooks::TD_FORCED),
            'headlineCookieOrigin' => _x('Who can set cookies on this website?', 'legal-text', Hooks::TD_FORCED),
            'headlineListOfServices' => _x('Which cookies are used on this website?', 'legal-text', Hooks::TD_FORCED),
            'diffToPrivacyPolicy' => _x(
                'This cookie policy is a supplement to this website\'s {{privacyPolicy}}privacy policy{{/privacyPolicy}}. The cookie policy specifies how a website uses cookies and what data it collects through them, while the privacy policy is a more comprehensive overview of all data processing activities on the website and beyond.',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'cookieTechnology' => join('<br /><br />', [
                _x(
                    'Cookies are small pieces of information similar to text files that can be stored and read on the end device. They serve various purposes, including maintaining basic functionalities of the website, security and privacy, providing optional functions of the website, collecting statistic data on visitor flows and providing marketing systems. Practical examples of what can be stored in cookies are the storage of login status in user accounts, the content of shopping baskets in e-commerce platforms, or a user ID for tracking behavior on the website.',
                    'legal-text',
                    Hooks::TD_FORCED
                ),
                _x(
                    'The information can technically be stored in various ways. The best-known examples of this are HTTP cookies and cookie-like technologies like local storage, session storage or IndexedDB. Each type of storage has different properties, which determine the technical handling, accessibility and controllers authorized to access the information. All these types of storage are usually summarized under the term "cookies" and are therefore called as such in this cookie policy.',
                    'legal-text',
                    Hooks::TD_FORCED
                ),
            ]),
            'legalBasisGdpr' => _x(
                'The setting and reading of cookies in the European Union (EU) and the European Economic Area (EEA) is in accordance with {{euLegalBasis}} only permitted if a user has given his consent on the basis of comprehensive information about the purposes of the processing. The website operator may also set cookies if they are strictly necessary to provide you as a user with the expressly requested service, e.g. the basic content of this website or other strictly necessary cookies for basic functionality of the website to be displayed to you without your consent.',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'legalBasisDsg' => _x(
                'The setting and reading of cookies in Switzerland is in accordance with Art. 45c (b) FMG (Fernmeldegesetz) only permitted if a user has the option to object the processing of data on the basis of comprehensive information about the purposes of the processing. In exceptional cases, consent must be obtained for the processing of personal data in cookies in Switzerland.',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'rightsOfTheVisitor' => sprintf(
                '%s<br /><ul><li>%s</li></ul>',
                _x(
                    'Services and their cookies can be set and read on the legal basis of your consent or a legitimate interest. When you first visited this website, you were asked for your consent and you had the opportunity to object to the use of certain services. We explained which legal basis is used for which service in our consent dialog.<br /><br />You have the right to view the history of your decisions, change your privacy settings, object to the use of services and revoke your consent at any time. Below you find possibilities to exercise your rights:',
                    'legal-text',
                    Hooks::TD_FORCED
                ),
                join('</li><li>', [
                    _x(
                        '[rcb-consent type="change" tag="a" text="Change privacy settings"]',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '[rcb-consent type="history" tag="a" text="Privacy settings history"]',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '[rcb-consent type="revoke" tag="a" text="Revoke consents" successmessage="You have successfully revoked consent for services with its cookies and personal data processing. The page will be reloaded now!"]',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                ])
            ),
            'manageCookies' => sprintf(
                '<ul><li>%s</li></li></ul><p>%s</p>',
                join('</li><li>', [
                    _x(
                        '<strong>How to remove cookies in a browser:</strong> To remove cookies from your device, you can clear the browsing data in the settings of your browser. This action will remove all cookies from all websites you\'ve visited, potentially including saved login details and site preferences. In some browsers, you can just delete the cookies and similar data without deleting the whole browsing history.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>How to control cookies in a browser:</strong> For more granular control over cookies specific to certain websites, navigate to the privacy and cookie settings within your used browser. Here, you can adjust preferences related to individual sites\' cookie usage.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>How to block cookies in a browser:</strong> It\'s possible to configure most modern browsers in their settings to block all cookies from being placed on your device. But blocking cookies could lead to certain services and functionalities not functioning correctly, e.g. logins as a user. You can also use extensions for many browsers which can block the setting of cookies on websites.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>How to manage cookies on this website:</strong> To tailor your preferences regarding cookies on this website, you can change your preferences at any time by clicking on the link in the section "Rights of the website visitor".',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                ]),
                _x(
                    'You can find more information on the handling of cookies <a href="https://devowl.io/rcb/cookie-handling/" target="_blank">devowl.io/rcb/cookie-handling/</a>.',
                    'legal-text',
                    Hooks::TD_FORCED
                )
            ),
            'typesOfCookies' => sprintf(
                '<ul><li>%s</li></li></ul>',
                join('</li><li>', [
                    _x(
                        '<strong>Essential cookies</strong> are indispensable for the basic functionality of a website and enable functions such as saving login information or access to secure areas of the site. These include, for example, security cookies, which can identify a bot that is not wanted on the website by storing an authentication key in a cookie.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>Functional cookies</strong> improve the user experience by storing information and preferences such as language settings or appearance settings of the website to provide personalized. This includes, for example, preference cookies that store the user\'s preferred video quality on a website.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>Statistics cookies</strong> collect data about the use of a website to gain insights into user behavior and optimize the performance of the site. This includes, for example, a randomly assigned user ID that makes it possible to attribute whether you have visited a subpage of the website.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                    _x(
                        '<strong>Marketing cookies</strong> are used to track user behavior across different websites and to display personalized advertising based on the interests and surfing behavior of users. This includes, for example, advertising cookies that store an interaction with an advertisement on the website and pass it on to advertisers or a chat system for pre-sale questions and customer support.',
                        'legal-text',
                        Hooks::TD_FORCED
                    ),
                ])
            ),
            'cookieOrigin' => _x(
                'On this website, both the operator of the website and third-party services which are used on this website may set cookies and access its content. Who can access which cookies and their content differs between first-party and third-party cookies. First-party cookies are created by the visited website and can only be read by the website operator and integrated third-party services on this website. Third-party cookies are set by usually third-party services on other domains, can be read on all websites on which the service that controls the domain is embedded and, for example, used by advertising networks to track user behavior across different websites and offer personalized advertising.',
                'legal-text',
                Hooks::TD_FORCED
            ),
            'additionalContent' => _x(
                'The cookie policy was last updated on {{dateOfUpdate}}.',
                'legal-text',
                Hooks::TD_FORCED
            ),
        ];
        $tempTd->teardown();
        return $defaults;
    }
}
