import { Divider } from "antd";
import { observer } from "mobx-react";
import { useState } from "react";

import { UpsellTag } from "@devowl-wp/react-cookie-banner-admin";

import { ConsentRecordBlocker } from "./blocker.js";
import { useStores } from "../../../../store/stores.js";
import { __, _i } from "../../../../utils/i18n.js";

import type { Consent } from "../../../../models/consent.js";
import type { FC } from "react";

const ConsentTableRecordPreview: FC<{ record: Consent; onPreview: (record: Consent, replay: boolean) => void }> =
    observer(({ record, onPreview }) => {
        const {
            optionStore: {
                others: { isPro },
            },
        } = useStores();

        const [replayBlockerRecord, setReplayBlockerRecord] = useState(false);

        const { custom_bypass, blocker, recorder, forwarded, referer, design_version } = record;

        return custom_bypass ? (
            <>
                <Divider>{__("Bypassed banner")}</Divider>
                <p className="description">
                    {_i(
                        __(
                            "There is no preview for this consent, as it was given implicitly by {{strong}}%s bypass{{/strong}}.",
                            record.custom_bypass_readable,
                        ),
                        {
                            strong: <strong />,
                        },
                    )}
                </p>
            </>
        ) : blocker > 0 ? (
            <>
                <Divider>{__("Content Blocker as at the time of consent")}</Divider>
                <p className="description">
                    {__(
                        "The consent to the service was given via a content blocker. Below you can see how the content blocker looked like when the user consented.",
                    )}
                </p>
                {isPro ? (
                    <ConsentRecordBlocker
                        record={record}
                        replayBlockerRecord={replayBlockerRecord}
                        replayFinished={() => setReplayBlockerRecord(false)}
                    />
                ) : (
                    <UpsellTag
                        title={__("You want to see what was displayed to the visitor?")}
                        tagText={__("Unlock preview")}
                        testDrive
                        feature="consent-preview-blocker"
                        assetName={__("pro-modal/consent-preview-blocker.png")}
                        description={__(
                            "We generate the content blocker that your visitor has seen from all settings, design preferences and services. You can see exactly how his or her consent was obtained and track clicks on buttons.",
                        )}
                    />
                )}
                {isPro && !!recorder && (
                    <>
                        <p className="description" style={{ marginTop: 15 }}>
                            {__(
                                "The process of how the website visitor interacted with the content blocker to give consent was recorded for documentation purposes. You can replay the interactions. Note that the dimensions of the content blocker and consent dialog do not have to be the same as the ones that were displayed to the website visitor, and the recording may not be fully accurate if, for example, you use custom CSS on your website or the visitor used a translating browser extension.",
                            )}
                        </p>
                        <button
                            className="button-primary button-large"
                            onClick={() => {
                                setReplayBlockerRecord(!replayBlockerRecord);
                            }}
                        >
                            {replayBlockerRecord ? __("Stop") : __("Replay interactions")}
                        </button>
                    </>
                )}
            </>
        ) : forwarded > 0 ? (
            <>
                <Divider>{__("Forwarded consent")}</Divider>
                <p className="description">
                    {_i(
                        __(
                            "There is no preview for this consent, as it was given implicitly by forwarding the consent. The user has given his/her consent via {{a}}%s{{/a}}, and this consent has been forwarded automatically.",
                            referer,
                        ),
                        {
                            a: <a href={referer} rel="noopener noreferrer" target="_blank" />,
                        },
                    )}
                </p>
            </>
        ) : (
            <>
                <Divider>{__("Cookie banner as at the time of consent")}</Divider>
                <p className="description">
                    {__(
                        "Use the button below to see what the cookie banner looked like at the time of the user's consent. The services/groups selected there also look the way the user saw them. A button framed in red shows which button the user clicked on at the time of consent.",
                    )}
                </p>
                {isPro ? (
                    <button
                        className="button-primary button-large"
                        onClick={() => {
                            onPreview(record, false);
                        }}
                        disabled={record.tcf_string && design_version < 7}
                    >
                        {__("Open banner")}
                    </button>
                ) : (
                    <UpsellTag
                        title={__("You want to see what was displayed to the visitor?")}
                        tagText={__("Unlock preview")}
                        testDrive
                        feature="consent-preview-banner"
                        assetName={__("pro-modal/consent-preview-banner.png")}
                        description={__(
                            "We generate the cookie banner from all (design) settings and services that the visitor has seen. You can see exactly how consent was obtained and track clicks on buttons.",
                        )}
                    />
                )}
                {isPro && !!recorder && (
                    <>
                        <p className="description" style={{ marginTop: 15 }}>
                            {__(
                                "The process of how the website visitor interacted with the cookie banner to give consent was recorded for documentation purposes. You can replay the interactions. Note that the recording may not be fully accurate if, for example, you use custom CSS on your website or the visitor used a translating browser extension.",
                            )}
                        </p>
                        <button
                            className="button-primary button-large"
                            onClick={() => {
                                onPreview(record, true);
                            }}
                            disabled={design_version < 6 || (record.tcf_string && design_version < 7)}
                        >
                            {__("Replay interactions")}
                        </button>
                        {design_version < 6 ? (
                            <div
                                className="notice notice-warning inline below-h2 notice-alt"
                                style={{ margin: "10px 0 0 0" }}
                            >
                                <p>
                                    {_i(
                                        __(
                                            "This consent was documented with a Real Cookie Banner version prior to 3.10.0. Due to rework to comply with the {{aAct}}European Accessibility Act{{/aAct}}, interactions with the currently installed version of Real Cookie Banner are no longer playable. However, they are still documented in the consent export. If you need to play the interactions, please request an old Real Cookie Banner version from the {{aSupport}}support{{/aSupport}}.",
                                        ),
                                        {
                                            aAct: (
                                                <a
                                                    href={__(
                                                        "https://eur-lex.europa.eu/legal-content/EN/TXT/?uri=CELEX:32019L0882",
                                                    )}
                                                    rel="noreferrer"
                                                    target="_blank"
                                                />
                                            ),
                                            aSupport: (
                                                <a
                                                    href={__("https://devowl.io/support/")}
                                                    rel="noreferrer"
                                                    target="_blank"
                                                />
                                            ),
                                        },
                                    )}
                                </p>
                            </div>
                        ) : record.tcf_string && design_version < 7 ? (
                            <div
                                className="notice notice-warning inline below-h2 notice-alt"
                                style={{ margin: "10px 0 0 0" }}
                            >
                                <p>
                                    {_i(
                                        __(
                                            "This consent was documented with a Real Cookie Banner version prior to 4.1.0. Due to rework to comply with the TCF 2.2 standard, interactions with the currently installed version of Real Cookie Banner are no longer playable. However, they are still documented in the consent export. If you need to play the interactions, please request an old Real Cookie Banner version from the {{aSupport}}support{{/aSupport}}.",
                                        ),
                                        {
                                            aSupport: (
                                                <a
                                                    href={__("https://devowl.io/support/")}
                                                    rel="noreferrer"
                                                    target="_blank"
                                                />
                                            ),
                                        },
                                    )}
                                </p>
                            </div>
                        ) : null}
                    </>
                )}
            </>
        );
    });

export { ConsentTableRecordPreview };
