import { App, Card, Col, Collapse, Row, Switch } from "antd";
import { observer } from "mobx-react";
import { useCallback } from "react";

import { LinkShortcodeForm } from "./linkForm.js";
import { ShortCodeMenuList } from "./menuList.js";
import { useConfigHintText } from "../../../../hooks/useConfigHintText.js";
import { useStores } from "../../../../store/stores.js";
import { __, _x } from "../../../../utils/i18n.js";
import { useProModalStickyLinks } from "../../../customize/controls/stickySwitcher.js";

import type { ComponentProps, FC } from "react";

const { Panel } = Collapse;

const ShortcodeCards: FC = observer(() => {
    const hintText = useConfigHintText("shortcodes");
    const { optionStore } = useStores();
    const {
        busySettings,
        isBannerStickyLinksEnabled,
        others: { customizeBannerUrl },
    } = optionStore;
    const { isPro, modal, tag } = useProModalStickyLinks();
    const { message } = App.useApp();

    const handleToggleBannerStickyLinks: ComponentProps<typeof Switch>["onChange"] = useCallback((status) => {
        optionStore.updateSettings({ isBannerStickyLinksEnabled: status }).then(() => {
            message.success(
                status
                    ? __("Sticky legal links are now enabled on your website")
                    : __("Sticky legal links are now disabled on your website"),
            );
        });
    }, []);

    return (
        <>
            {modal}
            <Collapse defaultActiveKey={["nav", "sticky"]} ghost>
                <Panel key="nav" header={<a>{__("Add links to existing menu")}</a>}>
                    <Card style={{ margin: 5 }}>
                        <ShortCodeMenuList />
                    </Card>
                </Panel>
                <Panel key="sticky" header={<a>{__("Sticky legal links widget")}</a>}>
                    <Card style={{ margin: 5 }}>
                        <Row align="middle">
                            <Col flex="auto">
                                <Switch
                                    loading={busySettings}
                                    onChange={handleToggleBannerStickyLinks}
                                    value={isPro ? isBannerStickyLinksEnabled : false}
                                    disabled={!isPro}
                                />
                                &nbsp;&nbsp;{__("Show a sticky legal links widget on the bottom of your website")}&nbsp;
                                {tag}
                                {!isPro && (
                                    <p className="description" style={{ marginTop: 10 }}>
                                        {__(
                                            "You can add the legal links in the free version of Real Cookie Banner as menu items or shortcodes (e.g. in your website footer).",
                                        )}
                                    </p>
                                )}
                            </Col>
                            {isPro && (
                                <Col>
                                    <a
                                        href={`${customizeBannerUrl.replace(
                                            /autofocus\[panel]=[\w-]+/,
                                            `autofocus[section]=real-cookie-banner-banner-sticky-links`,
                                        )}&return=${encodeURIComponent(window.location.href)}`}
                                        className="button button-primary"
                                    >
                                        {__("Customize")}
                                    </a>
                                </Col>
                            )}
                        </Row>
                    </Card>
                </Panel>
                <Panel key="shortcode" header={<a>{__("Generate shortcode (advanced)")}</a>}>
                    <Row>
                        <Col xl={8} sm={12} xs={24}>
                            <Card style={{ margin: 5 }} title={_x("Change privacy settings", "legal-text")}>
                                <LinkShortcodeForm type="change" />
                            </Card>
                        </Col>
                        <Col xl={8} sm={12} xs={24}>
                            <Card style={{ margin: 5 }} title={_x("Privacy settings history", "legal-text")}>
                                <LinkShortcodeForm type="history" />
                            </Card>
                        </Col>
                        <Col xl={8} sm={12} xs={24}>
                            <Card style={{ margin: 5 }} title={_x("Revoke consents", "legal-text")}>
                                <LinkShortcodeForm type="revoke" />
                            </Card>
                        </Col>
                    </Row>
                </Panel>
            </Collapse>
            <span
                className="description"
                style={{ display: "block", maxWidth: 800, margin: "30px auto 0", textAlign: "center" }}
            >
                {hintText}
            </span>
        </>
    );
});

export { ShortcodeCards };
