import { CheckOutlined } from "@ant-design/icons";
import { App, Empty, List, Tag } from "antd";
import { observer } from "mobx-react";
import { useCallback } from "react";

import { useStores } from "../../../../store/stores.js";
import { __ } from "../../../../utils/i18n.js";

import type { FC } from "react";

/**
 * List all available menus and allow to add the legal links via button.
 */
const ShortCodeMenuList: FC = observer(() => {
    const { message } = App.useApp();
    const { optionStore } = useStores();
    const {
        navMenus,
        busyCountryBypassUpdate,
        others: { adminUrl },
    } = optionStore;

    const handleAdd = useCallback(async (id: string) => {
        try {
            await optionStore.addLinksToNavigationMenu(id);
            message.success(__("Successfully added the links to your menu!"));
        } catch (e) {
            message.error((e as any).responseJSON.message);
        }
    }, []);

    return (
        <List<(typeof navMenus)[0]>
            loading={busyCountryBypassUpdate}
            itemLayout="horizontal"
            dataSource={navMenus}
            size="small"
            locale={{
                emptyText: (
                    <Empty description={__("No WordPress menu created yet.")}>
                        <a href={`${adminUrl}nav-menus.php`} className="button button-primary">
                            {__("Create menu")}
                        </a>
                    </Empty>
                ),
            }}
            renderItem={(row) => {
                const { id, applied, edit_url, languages, name } = row;
                return (
                    <List.Item
                        actions={[
                            applied ? (
                                <a key="apply">
                                    <CheckOutlined style={{ color: "#52c41a" }} />
                                    &nbsp;
                                    {__("Already added")}
                                </a>
                            ) : (
                                <a key="applied" onClick={() => handleAdd(id)}>
                                    {__("Add all legal links")}
                                </a>
                            ),
                            <a key="edit-manually" target="_blank" href={edit_url} rel="noreferrer">
                                {__("Edit manually")}
                            </a>,
                        ]}
                    >
                        <List.Item.Meta
                            title={
                                <span>
                                    {name} {languages.length > 0 && <Tag>{languages[0].language}</Tag>}
                                </span>
                            }
                            description={row.type === "legacy_nav" && Object.values(row.locations).join(", ")}
                        />
                    </List.Item>
                );
            }}
        />
    );
});

export { ShortCodeMenuList };
