import { CloseOutlined, EditOutlined } from "@ant-design/icons";
import { Popconfirm, Popover, Spin, Tooltip } from "antd";
import { observer } from "mobx-react";
import { useCallback, useState } from "react";
import { useNavigate } from "react-router-dom";

import { reactNl2Br } from "@devowl-wp/react-utils";

import { CookieGroupEditForm } from "./form.js";
import { useRouteCookieGroup } from "../../../../hooks/useRouteCookieGroup.js";
import { useStores } from "../../../../store/stores.js";
import { __ } from "../../../../utils/i18n.js";

import type { CookieGroupModel } from "../../../../models/cookieGroupModel.js";
import type { FC } from "react";

const CookieGroupTabHeadline: FC<{ group: CookieGroupModel }> = observer(({ group }) => {
    const {
        optionStore: {
            others: {
                hints: { deleteCookieGroup },
            },
        },
        cookieStore: { essentialGroup },
    } = useStores();
    const {
        busy,
        data: { id, name, description },
    } = group;
    const [isEditVisible, setIsEditVisible] = useState(false);
    const isActiveTab = useRouteCookieGroup().group.key === group.key;
    const isEssential = id === essentialGroup.key;
    const navigate = useNavigate();

    const handleDelete = useCallback(async () => {
        await group.delete({
            force: true,
        });
        navigate(`/cookies/${essentialGroup.key.toString()}`, { replace: true });
    }, [group, essentialGroup, isActiveTab]);

    return (
        <Spin spinning={busy}>
            <Tooltip title={description} placement="bottomLeft">
                <span>{name}</span>
            </Tooltip>
            {isActiveTab && (
                <Popover
                    open={isEditVisible}
                    arrow={{ pointAtCenter: true }}
                    content={<CookieGroupEditForm edit={group} onClose={() => setIsEditVisible(false)} />}
                    placement="bottomLeft"
                >
                    <EditOutlined onClick={() => setIsEditVisible(true)} style={{ margin: 0, marginLeft: 5 }} />
                </Popover>
            )}
            {isActiveTab &&
                (isEssential ? (
                    <Tooltip
                        placement="bottomLeft"
                        arrow={{ pointAtCenter: true }}
                        overlay={__(`The "%s" service group cannot be deleted.`, name)}
                    >
                        <CloseOutlined style={{ margin: 0, marginLeft: 5, color: "#d2d2d2" }} />
                    </Tooltip>
                ) : (
                    <Popconfirm
                        title={reactNl2Br(
                            [
                                __(
                                    "Are you sure you want to delete this service group? All services within this group will also be deleted. Consider moving them before you delete the service group.",
                                ),
                                ...deleteCookieGroup,
                            ].join("\n\n"),
                        )}
                        placement="bottom"
                        onConfirm={handleDelete}
                        okText={__("Delete")}
                        cancelText={__("Cancel")}
                        overlayStyle={{ maxWidth: 350 }}
                        onOpenChange={() => setIsEditVisible(false)}
                    >
                        <CloseOutlined style={{ margin: 0, marginLeft: 5 }} />
                    </Popconfirm>
                ))}
        </Spin>
    );
});

export { CookieGroupTabHeadline };
