import { CheckCircleOutlined, MinusCircleOutlined } from "@ant-design/icons";
import { Space, Spin, Tag } from "antd";
import { observer } from "mobx-react";
import { useState } from "react";

import { DashboardChecklistItem } from "./checklistItem.js";
import { useStores } from "../../../store/stores.js";
import { __ } from "../../../utils/i18n.js";

import type { ComponentProps, FC } from "react";

const DashboardChecklistCardContent: FC<{
    showHidden?: boolean;
    onLinkClick?: ComponentProps<typeof DashboardChecklistItem>["onLinkClick"];
}> = observer(({ onLinkClick, ...props }) => {
    const { checklistStore } = useStores();
    const {
        busyChecklist,
        items,
        completed,
        checkable,
        done,
        checklist: { overdue },
    } = checklistStore;
    const [showHidden, setShowHidden] = useState(
        typeof props.showHidden === "boolean" ? props.showHidden : done ? false : true,
    );

    return (
        <Spin spinning={busyChecklist}>
            <Space direction="vertical" size="large">
                {overdue && (
                    <div className="notice notice-warning inline below-h2 notice-alt" style={{ margin: 0 }}>
                        <p>
                            {__(
                                "How time flies! You have installed Real Cookie Banner some time ago, but you still haven't finished the configuration yet.",
                            )}
                        </p>
                    </div>
                )}
                <p className="description">
                    {__(
                        "We have collected all relevant steps for the legally compliant use of Real Cookie Banner after the first installation. Set up the cookie banner step by step to add a cookie banner to your website in compliance with the ePrivacy Directive and GDPR.",
                    )}
                </p>
                <div>
                    {done &&
                        (showHidden ? (
                            <Tag
                                icon={<MinusCircleOutlined />}
                                color="default"
                                style={{ cursor: "pointer" }}
                                onClick={() => setShowHidden(false)}
                            >
                                {__("Hide completed steps")}
                            </Tag>
                        ) : (
                            <Tag
                                icon={<CheckCircleOutlined />}
                                color="success"
                                style={{ cursor: "pointer" }}
                                onClick={() => setShowHidden(true)}
                            >
                                {__(
                                    "%d / %d steps completed",
                                    completed.length > checkable.length ? checkable.length : completed.length, // avoid overflow due infos
                                    checkable.length,
                                )}
                            </Tag>
                        ))}
                    {items.map(({ id, ...rest }) => {
                        if (!showHidden && rest.checked) {
                            return null;
                        }

                        return (
                            <DashboardChecklistItem
                                key={id}
                                id={id}
                                {...rest}
                                onLinkClick={(e) => {
                                    if (["get-pro"].indexOf(id) > -1) {
                                        checklistStore.toggleChecklistItem(id, true);
                                    }
                                    onLinkClick?.(e);
                                }}
                            />
                        );
                    })}
                </div>
            </Space>
        </Spin>
    );
});

export { DashboardChecklistCardContent };
