import { Popconfirm } from "antd";
import { observer } from "mobx-react";
import { useLocation } from "react-router-dom";

import { Notices } from "@devowl-wp/react-cookie-banner-admin";

import { useStores } from "../../store/stores.js";
import { __, _i } from "../../utils/i18n.js";

import type { FC } from "react";

const NoticeBannerlessConsent: FC = observer(() => {
    const { optionStore, cookieStore } = useStores();
    const {
        bannerlessConsentChecks: { legalBasisLegitimateInterest, legalBasisConsentWithoutVisualContentBlocker },
        isBannerLessConsent,
    } = optionStore;
    const { pathname } = useLocation();

    return (
        <Notices
            notices={[
                {
                    nop: true,
                    key: "bannerless-consent-legitimate-interest",
                    message: isBannerLessConsent &&
                        pathname.indexOf("/edit/") === -1 &&
                        pathname.indexOf("/new") === -1 &&
                        legalBasisLegitimateInterest.length > 0 && (
                            <>
                                <p>
                                    {__(
                                        "You have decided to obtain consent without cookie banners (banner-less consent) on your website. This means that when your website visitors visit your website for the first time, they will not know that you use this service on the basis of a legitimate interest and can object to it. They will need to read your cookie policy or privacy policy to find out.",
                                    )}
                                </p>
                                <p>
                                    {__(
                                        "At the same time, you use the following non-essential services on the legal basis of a legitimate interest:",
                                    )}
                                </p>
                                <ul>
                                    {legalBasisLegitimateInterest.map(({ name, groupId, id }) => (
                                        <li key={id}>
                                            <strong>{name}</strong>&nbsp;&bull;&nbsp;
                                            <a href={`#/cookies/${groupId}/edit/${id}`}>{__("Edit")}</a>
                                        </li>
                                    ))}
                                </ul>
                                <p>
                                    {_i(
                                        __(
                                            "Check whether there really is a legitimate interest in the legal sense according to {{a}}Art. 6 (1) (f) GDPR{{/a}}, which is only given in rare cases!",
                                        ),
                                        {
                                            a: (
                                                <a
                                                    href="https://gdpr-text.com/read/article-6/"
                                                    target="_blank"
                                                    rel="noreferrer"
                                                />
                                            ),
                                        },
                                    )}
                                </p>
                                <p>
                                    <a onClick={() => optionStore.dismissBannerlessConsentLegitimateServicesNotice()}>
                                        {__("Dismiss notice")}
                                    </a>
                                </p>
                            </>
                        ),
                    severity: "warning",
                },
                {
                    nop: true,
                    key: "bannerless-consent-services-without-visual-content-blocker",
                    message: isBannerLessConsent &&
                        pathname.indexOf("/edit/") === -1 &&
                        pathname.indexOf("/new") === -1 &&
                        legalBasisConsentWithoutVisualContentBlocker.length > 0 && (
                            <>
                                <p>
                                    {__(
                                        "You have decided to obtain consent without cookie banners (banner-less consent) on your website. At the same time, some non-essential services, which requires a consent, doese not have visual content blockers, which is why website visitors without cookie banners cannot consent to them (except via the legal link to change the privacy settings).",
                                    )}
                                </p>
                                <p>
                                    {__(
                                        "You should create visual content blockers for these services, if the services display visual elements, or disable them for your website! The following services are affected:",
                                    )}
                                </p>
                                <ul>
                                    {legalBasisConsentWithoutVisualContentBlocker.map(({ name, groupId, id }) => (
                                        <li key={id}>
                                            <strong>{name}</strong>&nbsp;&bull;&nbsp;
                                            <a href={`#/blocker/new`}>{__("Create Content Blocker")}</a>
                                            &nbsp;&bull;&nbsp;
                                            <Popconfirm
                                                title={__("Are you sure you want to disable this service?")}
                                                onConfirm={async () => {
                                                    const group = cookieStore.groups.entries.get(groupId);
                                                    await group.fetchCookies();
                                                    const cookie = group.cookies.entries.get(id);
                                                    cookie.setStatus("private");
                                                    return cookie.patch();
                                                }}
                                                okText={__("Deactivate")}
                                                cancelText={__("Cancel")}
                                            >
                                                <a>{__("Disable")}</a>
                                            </Popconfirm>
                                        </li>
                                    ))}
                                </ul>
                                <p>
                                    <a
                                        onClick={() =>
                                            optionStore.dismissBannerlessConsentServicesWithoutVisualContentBlockerNotice()
                                        }
                                    >
                                        {__("Dismiss notice")}
                                    </a>
                                </p>
                            </>
                        ),
                    severity: "warning",
                },
            ]}
        />
    );
});

export { NoticeBannerlessConsent };
