import { Popover } from "antd";
import { observer } from "mobx-react";
import { useCallback, useState } from "react";

import { useCookieExpertsModal } from "@devowl-wp/react-cookie-banner-admin";

import { useRouteBlocker } from "../../../hooks/useRouteBlocker.js";
import { useStores } from "../../../store/stores.js";
import { __, _i } from "../../../utils/i18n.js";

import type { ScannerResultExternalUrl } from "../../../models/scannerResultExternalUrl.js";
import type { FC } from "react";

const ScannerExternalUrlActionPopover: FC<{ item: ScannerResultExternalUrl }> = observer(({ item }) => {
    const [isVisible, setIsVisible] = useState(false);
    const {
        cookieStore: { essentialGroup },
    } = useStores();
    const { openDialog } = useCookieExpertsModal();
    const {
        data: { host },
    } = item;
    const { addLink: addBlockerLink } = useRouteBlocker();

    const toggleVisible = useCallback(() => {
        setIsVisible(!isVisible);
    }, [isVisible]);

    const openCookieExperts = useCallback(() => {
        setIsVisible(false);
        openDialog();
    }, [openDialog]);

    // Always navigate back to the scanner results
    const navigateAfterCreation = `navigateAfterCreation=${encodeURIComponent(window.location.href)}`;

    return (
        <Popover
            open={isVisible}
            content={
                <>
                    <p>
                        <strong>
                            {_i(
                                __(
                                    "Does {{i}}%s{{/i}} belong to an essential service without which your website would technically no longer work?",
                                    host,
                                ),
                                {
                                    i: <i />,
                                },
                            )}
                        </strong>
                    </p>
                    <p>
                        <a
                            className="button button-primary"
                            href={`${addBlockerLink}?force=scratch&attributes=${JSON.stringify({
                                rules: `*${host}*`,
                            })}&${navigateAfterCreation}`}
                        >
                            {__("No")}
                        </a>
                        &nbsp;
                        <a
                            className="button"
                            href={`#/cookies/${essentialGroup?.key}/new?force=scratch&${navigateAfterCreation}`}
                        >
                            {__("Yes")}
                        </a>
                        &nbsp;
                        <button className="button" onClick={openCookieExperts}>
                            {__("I do not know")}
                        </button>
                    </p>
                    <p className="description">
                        {_i(
                            __(
                                "{{strong}}No:{{/strong}} Non-essential services that process personal data (e.g. IP address in some countries) or set cookies may only be loaded after consent. You should block them using a content blocker until consent is given. {{i}}Examples would be iframes, YouTube and similar embeddings or tracking tools.{{/i}}",
                            ),
                            { strong: <strong />, i: <i /> },
                        )}
                    </p>
                    <p className="description">
                        {_i(
                            __(
                                "{{strong}}Yes (rare cases):{{/strong}} You should inform your users about the use of the service in the essential services group. You do not need to create a content blocker, as the service can be loaded without consent. {{i}}Examples are privacy-friendly spam protection tools or security tools.{{/i}}",
                            ),
                            { strong: <strong />, i: <i /> },
                        )}
                    </p>
                    <p>
                        <button className="button" onClick={toggleVisible}>
                            {__("Close")}
                        </button>
                    </p>
                </>
            }
            placement="right"
            overlayStyle={{ maxWidth: 350 }}
        >
            <a onClick={toggleVisible}>{__("Handle external URL")}</a>
        </Popover>
    );
});

export { ScannerExternalUrlActionPopover };
