import { useMemo } from "react";

import { A11yNotice, useA11yDispatcher, useCustomizeValues } from "@devowl-wp/customize";

import { __ } from "../../utils/i18n.js";

import type { FC } from "react";

const CustomizeA11yMobileExperience: FC<{
    /**
     * List of setting IDs. The first one needs to be the settings which expose a
     * the mobile experience scaling factor in percent, and all other settings all available font sizes.
     *
     * This property does not listen to changes of the array and is used one-time.
     */
    settings: string[];
}> = ({ settings }) => {
    const values = useCustomizeValues(settings);

    const hasError = useMemo(() => {
        const [{ value: percent }, ...fontSizes] = Object.values(values);
        const smallestFontSize = Math.min(...fontSizes.filter(({ visible }) => visible).map(Number));
        const recommendedPercent = smallestFontSize <= 10 ? 100 : smallestFontSize <= 12 ? 93 : 90;
        return +percent < recommendedPercent ? recommendedPercent : false;
    }, [JSON.stringify(values)]);

    useA11yDispatcher({
        setting: settings[0],
        valid: !hasError,
    });

    if (hasError === false) {
        return null;
    }

    return (
        <A11yNotice
            severity="warning"
            text={__(
                'We recommend a scaling factor of at least %d %% in order to comply with the {{aPerc}}"perceptibility" of the WCAG 2.2 standard{{/aPerc}} and thereby meet the {{aEuMinimum}}EU\'s minimum legal requirements for accessibility{{/aEuMinimum}}.',
                hasError,
            )}
        />
    );
};

export { CustomizeA11yMobileExperience };
