import { useMemo } from "react";

import { A11yNotice, useA11yDispatcher, useCustomizeValues } from "@devowl-wp/customize";
import type { CustomizeLayout } from "@devowl-wp/react-cookie-banner";

import { __ } from "../../utils/i18n.js";

import type { FC } from "react";

const CustomizeA11yOverlay: FC<{
    /**
     * List of setting IDs. The first three needs to be the settings which expose a
     * dialog type (banner, dialog), block content until consent (boolean) and overlay opacity.
     *
     * This property does not listen to changes of the array and is used one-time.
     */
    settings: string[];
}> = ({ settings }) => {
    const values = useCustomizeValues(settings);

    const hasError = useMemo(() => {
        const {
            [settings[0]]: { value: type },
            [settings[1]]: { value: overlay },
            [settings[2]]: { value: overlayBgAlpha },
        } = values;

        if ((type as CustomizeLayout["type"]) === "banner") {
            return false;
        }

        return !(overlay as CustomizeLayout["overlay"]) || (overlayBgAlpha as CustomizeLayout["overlayBgAlpha"]) < 50;
    }, [JSON.stringify(values)]);

    useA11yDispatcher({
        setting: settings[1],
        valid: !hasError,
    });

    if (hasError === false) {
        return null;
    }

    return (
        <A11yNotice
            severity="warning"
            text={__(
                'We recommend to show an overlay until consent and an overlay opacity over 50 %% to comply with the {{aPerc}}"perceptibility" of the WCAG 2.2 standard{{/aPerc}} and thereby meet the {{aEuMinimum}}EU\'s minimum legal requirements for accessibility{{/aEuMinimum}}.',
            )}
        />
    );
};

export { CustomizeA11yOverlay };
