import { useEffect, useState } from "react";

import { CUSTOM_CONTROL_INPUT_CHECKBOX_CLASS_NAME, listenPanelExpanded, useCustomControl } from "@devowl-wp/customize";
import { useUpsellModal } from "@devowl-wp/react-cookie-banner-admin";
import { usePlainCss } from "@devowl-wp/react-utils";

import { useStores } from "../../../store/stores.js";
import { __ } from "../../../utils/i18n.js";

import type { FC } from "react";

type MobileSwitcherControlProps = Record<string, never>;

function useProModalMobileExperience() {
    return useUpsellModal({
        title: __("Want to optimize the cookie banner for mobile users?"),
        testDrive: true,
        feature: "mobile-experience",
        assetName: __("pro-modal/mobile-optimization.png"),
        description: __(
            "Cookie banners are a necessary evil that takes up a lot of space, especially on smartphones. With mobile optimization you can customize the cookie banner so that it is more discreet and at the same time easy to read on smartphones.",
        ),
    });
}

const MobileSwitcherControl: FC = () => {
    const {
        optionStore: {
            others: {
                customizeIdsBanner: {
                    sections: { mobile: mobileSectionId },
                },
            },
        },
    } = useStores();
    const { value, setValue } = useCustomControl<boolean, MobileSwitcherControlProps>();
    const [selectedDimensions, setSelectedDimensions] = useState("375,812");
    const [isSectionOpen, setIsSectionOpen] = useState(false);
    const { isPro, modal, tag } = useProModalMobileExperience();

    useEffect(() => {
        listenPanelExpanded(mobileSectionId, setIsSectionOpen);
    }, []);

    const [deviceWidth, deviceHeight] = selectedDimensions.split(",").map(Number);
    usePlainCss(
        value && isSectionOpen
            ? `#customize-preview {
    width: ${deviceWidth}px !important;
    height: ${deviceHeight}px !important;
    margin: auto 0 auto calc(${deviceWidth}px / 2 * -1) !important;
    max-height: 100%;
    max-width: 100%;
    left: 50%;
}

#customize-footer-actions .devices-wrapper {
    display: none;
}`
            : "",
    );

    return (
        <>
            {modal}
            <label>
                <input
                    className={CUSTOM_CONTROL_INPUT_CHECKBOX_CLASS_NAME}
                    type="checkbox"
                    disabled={!isPro}
                    value="1"
                    checked={isPro ? value : false}
                    onChange={() => setValue(!value)}
                />
                {__("Enable mobile optimization")} {tag}
            </label>
            <span className="description customize-control-description" style={{ marginTop: 5 }}>
                {__(
                    "To ensure that the cookie banner does not take up too much space on mobile devices, its display can be reduced for this class of device.",
                )}
            </span>
            {value && (
                <>
                    <span className="customize-control-title" style={{ marginTop: 10 }}>
                        {__("Simulate device in preview")}
                    </span>
                    <select value={selectedDimensions} onChange={(e) => setSelectedDimensions(e.target.value)}>
                        <option value="411,731">Pixel 2</option>
                        <option value="411,823">Pixel 2 XL</option>
                        <option value="320,568">iPhone 5/SE</option>
                        <option value="375,667">iPhone 6/7/8</option>
                        <option value="414,736">iPhone 6/7/8 Plus</option>
                        <option value="375,812">iPhone X</option>
                    </select>
                </>
            )}
        </>
    );
};

export { MobileSwitcherControl, useProModalMobileExperience };
