import "../../style/controls/presetSelector.scss";

import { Tag } from "antd";
import { observer } from "mobx-react";
import { useCallback, useState } from "react";

import { UPSELL_TAG_BACKGROUND_COLOR, useUpsellModal } from "@devowl-wp/react-cookie-banner-admin";

import { INTERVAL_SWITCH_INDIVIDUAL_PRIVACY_SECONDS } from "./presetSelector.js";
import { BannerPreset } from "../../models/bannerPreset.js";
import { __ } from "../../utils/i18n.js";

import type { FC } from "react";

let renderTimeout: ReturnType<typeof setTimeout>;
let individualPrivacyInterval: ReturnType<typeof setInterval>;

const CustomizePresetCard: FC<{ preset: BannerPreset; onApply?: () => void }> = observer(({ preset, onApply }) => {
    const [applied, setApplied] = useState(false);
    const {
        isPro,
        open: openProModal,
        modal,
    } = useUpsellModal({
        title: __("Want to use %s preset?", preset.name),
        testDrive: true,
        feature: "customize-preset",
        description: __(
            "Only a limited number of presets are available for the free version of Real Cookie Banner. Start now with the PRO version and create a design from a preset with just one click.",
        ),
    });
    const iframeStore = BannerPreset.getIframeStore().customizeBannerStore;

    const handleMouseEnter = useCallback(() => {
        clearTimeout(renderTimeout);
        clearInterval(individualPrivacyInterval);
        renderTimeout = setTimeout(() => {
            setApplied(false);
            preset.previewInUi();

            // Toggle individual privacy preference after x seconds
            individualPrivacyInterval = setInterval(() => {
                iframeStore.setIndividualPrivacyOpen(!iframeStore.individualPrivacyOpen);
            }, INTERVAL_SWITCH_INDIVIDUAL_PRIVACY_SECONDS * 1000);
        }, 100);
    }, [setApplied, preset, iframeStore]);

    const handleMouseLeave = useCallback(() => {
        clearTimeout(renderTimeout);
        clearInterval(individualPrivacyInterval);
        iframeStore.setIndividualPrivacyOpen(false);
        if (!applied) {
            renderTimeout = setTimeout(() => {
                preset.resetPreviewInUi();
            }, 100);
        }
    }, [preset, applied]);

    const handleClick = useCallback(() => {
        clearInterval(individualPrivacyInterval);
        if (!preset.applyInUi()) {
            // Can currently only return `false` when not in Pro version
            openProModal();
        } else {
            setApplied(true);
            onApply?.();
        }
    }, [setApplied, preset]);

    return (
        <div className="customize-rcb-preset" onMouseEnter={handleMouseEnter} onMouseLeave={handleMouseLeave}>
            {modal}
            <strong>
                {preset.name} {!isPro && preset.needsPro && <Tag color={UPSELL_TAG_BACKGROUND_COLOR}>PRO</Tag>}
                {preset.tags?.map(([color, text]) => (
                    <Tag key={text} color={color}>
                        {text}
                    </Tag>
                ))}
            </strong>
            <span>{preset.description}</span>
            <div className="btn-apply hidden" onClick={handleClick}>
                {applied ? __("Applied!") : __("Apply")}
            </div>
        </div>
    );
});

export { CustomizePresetCard };
