import { observer } from "mobx-react";

import type { BannerContext } from "@devowl-wp/react-cookie-banner";
import { Banner, useBannerProvider, useTcf } from "@devowl-wp/react-cookie-banner";
import { YieldLazyLoad } from "@devowl-wp/react-utils";

import { getCookieConsentManager } from "../others/getCookieConsentManager.js";
import { useStores } from "../store/stores.js";

import type { FC } from "react";

/**
 * Simply connects a mobx store to the banner itself. Usually, the visitor
 * does not get any store, a simply context is used instead (see `banner.tsx`).
 */
const CustomizeBanner: FC<{ poweredLink: BannerContext["contextValue"]["poweredLink"] }> = observer(
    ({ poweredLink }) => {
        const { optionStore, customizeBannerStore } = useStores();
        const {
            others: {
                customizeValuesBanner: {
                    layout,
                    decision,
                    design,
                    headerDesign,
                    bodyDesign,
                    footerDesign,
                    texts,
                    individualLayout,
                    saveButton,
                    group,
                    individualTexts,
                    customCss,
                    mobile,
                    sticky,
                },
                frontend,
                bannerDesignVersion,
                iso3166OneAlpha2,
                bannerI18n,
                pageRequestUuid4,
                isPoweredByLinkDisabledByException,
                affiliate,
            },
        } = optionStore;
        const { visible, individualPrivacyOpen, previewCheckboxActiveState, previewStickyMenuOpenState } =
            customizeBannerStore;

        // TCF compatibility
        const tcfObject = useTcf(
            frontend.isTcf,
            frontend.tcf,
            frontend.tcfMetadata,
            getCookieConsentManager().getOptions(),
        );

        const [BannerContextProvider, bannerContextValue] = useBannerProvider(
            {
                layout: {
                    ...layout,
                    // In customizer preview, always show animation independent of device
                    animationInOnlyMobile: false,
                    animationOutOnlyMobile: false,
                },
                decision: { ...decision },
                design: { ...design },
                headerDesign: { ...headerDesign },
                bodyDesign: { ...bodyDesign },
                footerDesign: {
                    ...footerDesign,
                    ...{ poweredByLink: isPoweredByLinkDisabledByException ? false : footerDesign.poweredByLink },
                },
                texts: { ...texts },
                individualLayout: { ...individualLayout },
                saveButton: { ...saveButton },
                group: { ...group },
                individualTexts: { ...individualTexts },
                customCss: { ...customCss },
                mobile: { ...mobile },
                sticky: { ...sticky },

                ...frontend,
                // `frontend` also keeps an array of `blocker` which has a conflict with the Blocker context
                // -> we need to exclude this property in the context as the cookie banner should not have any
                // `blocker` property.
                ...{ blocker: undefined },

                isConsentGiven: true,
                pageRequestUuid4,
                poweredLink,
                iso3166OneAlpha2,
                gcmConsent: [],
                tcf: undefined,
                tcfFilterBy: "legInt",
                visible,
                individualPrivacyOpen,
                previewCheckboxActiveState,
                previewStickyMenuOpenState,
                designVersion: bannerDesignVersion,
                i18n: bannerI18n,
                affiliate,
                consent: {},
                didGroupFirstChange: false,
                onPersistConsent: async () => {},
                onApplyConsent: async () => {},
                suspense: { tcf: tcfObject },
            },
            {
                onSave: () => {},
                updateGroupChecked: () => {},
                updateCookieChecked: () => {},
                updateGcmConsentTypeChecked: () => {},
            },
            {
                inherit: [
                    "visible",
                    "previewCheckboxActiveState",
                    "previewStickyMenuOpenState",
                    "individualPrivacyOpen",
                    "layout",
                    "decision",
                    "design",
                    "headerDesign",
                    "bodyDesign",
                    "footerDesign",
                    "texts",
                    "individualLayout",
                    "saveButton",
                    "group",
                    "individualTexts",
                    "customCss",
                    "mobile",
                    "sticky",
                ],
            },
        );

        return (
            <BannerContextProvider value={bannerContextValue}>
                <YieldLazyLoad promise={tcfObject}>
                    <Banner />
                </YieldLazyLoad>
            </BannerContextProvider>
        );
    },
);

export { CustomizeBanner };
