import { CheckCircleTwoTone, CloseCircleTwoTone } from "@ant-design/icons";
import { Col, Divider, Row } from "antd";
import moment from "moment";

import { LearnMoreTag, useCookieExpertsModal } from "@devowl-wp/react-cookie-banner-admin";

import { useStores } from "../store/stores.js";
import { __, _i } from "../utils/i18n.js";

import type { ReactNode } from "react";

type ConfigHintTextIdentifier =
    | "cookie"
    | "blocker"
    | "list-of-consents"
    | "shortcodes"
    | "tcf-vendor"
    | "tcf-integration"
    | "scanner"
    | "import"
    | "consents-deleted";

function useConfigHintText(identifier: ConfigHintTextIdentifier, ...sprintfArgs: string[]): ReactNode {
    const {
        optionStore: { isTcf, consentsDeletedAt, consentDuration },
    } = useStores();

    switch (identifier) {
        case "scanner": {
            const { openDialog } = useCookieExpertsModal();
            return (
                <>
                    <p className="description">
                        {__(
                            "The scanner finds services that you use on your website that might set/read cookies or process personal data. This is e.g. Google Analytics, YouTube or Elementor. If there is no template for a service, you will see from which external URLs content, scripts etc. are embedded. This allows you to set up your cookie banner quickly and easily.",
                        )}
                    </p>
                    <p className="description">
                        {_i(
                            __(
                                "We explicitly do not find cookies because that would not work reliably. {{a}}We explained why in our knowledge base.{{/a}}",
                            ),
                            {
                                a: (
                                    <a
                                        rel="noreferrer"
                                        href={__(
                                            "https://devowl.io/knowledge-base/real-cookie-banner-cookie-scanner-finds-cookies-automatically/",
                                        )}
                                        target="_blank"
                                    />
                                ),
                            },
                        )}
                    </p>
                    <Row style={{ margin: "10px 0" }}>
                        <Col span={11}>
                            <div style={{ paddingRight: 10 }}>
                                <Divider>{__("What the scanner finds ...")}</Divider>
                                {[
                                    __("External services (with and without template)"),
                                    __("WordPress plugins with templates that require consent"),
                                    __("Automatic check of all subpages of your website"),
                                ].map((text, i) => (
                                    <div key={i} style={{ marginBottom: 10 }}>
                                        <CheckCircleTwoTone twoToneColor="#52c41a" />
                                        &nbsp;&nbsp;
                                        {text}
                                    </div>
                                ))}
                            </div>
                        </Col>
                        <Col span={2} style={{ textAlign: "center" }}>
                            <Divider type="vertical" style={{ height: "100%" }} />
                        </Col>
                        <Col span={11}>
                            <div>
                                <Divider>{__("... and what it does not")}</Divider>
                                {[
                                    __("Cookies from unknown WordPress plugins"),
                                    __("Services embedded after the page load via JavaScript"),
                                    __("Complete coverage of your individual use case"),
                                ].map((text, i) => (
                                    <div key={i} style={{ marginBottom: 10 }}>
                                        <CloseCircleTwoTone twoToneColor="#eb2f96" />
                                        &nbsp;&nbsp;
                                        {text}
                                    </div>
                                ))}
                            </div>
                        </Col>
                    </Row>
                    <p className="description">
                        {_i(
                            __(
                                "Just by using the scanner, you will not set up your cookie banner one hundred percent correctly. If it is too complex or time-consuming for you to set up the cookie banner yourself, just let one of our {{a}}cookie experts{{/a}} set it up for you!",
                            ),
                            {
                                a: <a onClick={openDialog} />,
                            },
                        )}
                    </p>
                </>
            );
        }
        case "cookie":
            return (
                <>
                    {__(
                        "What are services? Services can be external applications such as Google Analytics or WordPress plugins or themes that process personal data (e.g. IP address) and/or set cookies. Cookies (and similar technologies) are small text files that are stored on the device of visitors to your website. You can store information about the visitor in cookies, such as the website's language, or unique advertising IDs to display personalized advertising. You, as the site owner, must ensure that cookies are only placed on your visitors' devices and personal data are only processed if they have given their explicit consent. Unless you have a legitimate interest in the legal sense to do so even without consent. You can define here all the services you use and their cookies with their legal and technical information.",
                    )}
                    &nbsp;
                    <LearnMoreTag url={__("https://devowl.io/cookies/definition-cookies/")} />
                </>
            );
        case "blocker":
            return __(
                "What is a content blocker? Imagine that a user of your website does not accept all services. At the same time, you have integrated e.g. a YouTube video that sets cookies that the visitor has not agreed to. According to the ePrivacy Directive, this is prohibited. Content blockers automatically replace iframes, script and link tags like YouTube videos for such users and offer them to watch the video as soon as they agree to load it.",
            );
        case "list-of-consents":
            return _i(
                __(
                    "Consents are automatically documented in order to be able to prove compliance with the legal requirements according to {{a}}Art. 5 GDPR{{/a}} and, in case of dispute, to prove how the consent was obtained.",
                ),
                {
                    a: <a href={__("https://gdpr-text.com/read/article-5/")} target="_blank" rel="noreferrer" />,
                },
            );
        case "consents-deleted":
            return (
                <>
                    {consentsDeletedAt
                        ? __(
                              "Consents before %s has been automatically deleted according to the settings you have made.",
                              moment(consentsDeletedAt)
                                  .subtract(consentDuration, "months")
                                  .toDate()
                                  .toLocaleString(document.documentElement.lang),
                          )
                        : null}
                </>
            );
        case "shortcodes":
            return (
                <>
                    {_i(
                        __(
                            "Your website visitors must be able to view their consent history, change their consent, or withdraw their consent at any time. This must be as easy as giving consent. Therefore, the legal links must be included on every subpage of the website (e.g. in the footer or as a floating icon called sticky legal links widget).",
                        ),
                        {
                            strong: <strong />,
                        },
                    )}
                    <br />
                    <br />
                    {_i(
                        __(
                            "Hiding these options, e.g. in the privacy policy, is in the opinion of multiple data protection authorities in the EU a violation of the GDPR.",
                        ),
                        {
                            a: (
                                <a
                                    href={__("https://devowl.io/ger-dsk-orientierungshilfe-cookie-banner")}
                                    target="_blank"
                                    rel="noreferrer"
                                />
                            ),
                        },
                    )}
                    {isTcf ? (
                        <>
                            <br />
                            <br />
                            {__(
                                "To meet the requirements of the TCF standard, the shortcodes should be placed near the link to the privacy policy.",
                            )}
                        </>
                    ) : null}
                </>
            );
        case "tcf-vendor":
            return _i(
                __(
                    "What is a TCF vendor? According to the IAB Europe Transparency & Consent Framework (TCF), any service (e.g. Google for Google Ads) that wants to use consents according to the TCF standard must register as a vendor in the {{a}}Global Vendor List (GVL){{/a}}. All TCF vendors specify for which purposes they need consent to process data and set cookies and which features they can offer with these consents. They also provide a link to their privacy policy for further information. You, as a website operator, must obtain consent in your cookie banner for all vendors you work with. You can limit the requested purposes of vendors to keep consents as privacy-friendly as possible.",
                ),
                {
                    a: <a href={__("https://iabeurope.eu/vendor-list-tcf/")} target="_blank" rel="noreferrer" />,
                },
            );
        case "tcf-integration":
            return (
                <>
                    {__(
                        "You are trying to obtain consent for the advertising network %s. In order to be able to display advertising, you must obtain consent in accordance with the TCF standard. Therefore, please first activate TCF compatibility for your cookie banner so that you can then make all the necessary configurations!",
                        ...sprintfArgs,
                    )}
                </>
            );
        case "import":
            return (
                <>
                    {__(
                        "You can export and import all or only some of the settings you made in Real Cookie Banner. If you have several websites, you can save a lot of time by transferring the settings comfortably.",
                    )}
                    <br />
                    <br />
                    {__("Also, you can export documented consents to save them in a local backup.")}
                </>
            );
        default:
            return "";
    }
}

export { useConfigHintText, type ConfigHintTextIdentifier };
