import { createRoot } from "react-dom/client";

import {
    HeadlineBeforeSection,
    a11yConstrastRatio,
    a11yFontSize,
    a11yFontWeight,
    a11yTextDecoration,
    getSidebarCustomize,
    sidebarFactory,
} from "@devowl-wp/customize";
import type { Customize } from "@devowl-wp/react-cookie-banner";

import { applyCustomizeProTag } from "./applyCustomizeProTag.js";
import { CustomizeA11yMobileExperience } from "../components/customize/a11yMobileExperience.js";
import { CustomizeA11yOverlay } from "../components/customize/a11yOverlay.js";
import { CustomizeA11yScore } from "../components/customize/a11yScore.js";
import { CustomizeContextProvider } from "../components/customize/contextProvider.js";
import { DecisionButtonTypeSelectControl } from "../components/customize/controls/decisionButtonTypeSelectControl.js";
import { GroupsFirstViewControl } from "../components/customize/controls/groupsFirstView.js";
import { MobileSwitcherControl } from "../components/customize/controls/mobileSwitcher.js";
import { StickyLinksControl } from "../components/customize/controls/stickySwitcher.js";
import { CustomizePresetSelector } from "../components/customize/presetSelector.js";
import { RootStore } from "../store/stores.js";
import { __, _i } from "../utils/i18n.js";

const MIN_RATIO = 4.5;
const MIN_TCF_RATIO = 5;

/**
 * Apply common callbacks in customize banner sidebar.
 */
function applyCustomizeBannerSidebar() {
    const {
        optionStore: {
            others: {
                isPro,
                customizeIdsBanner,
                customizeDefaultsBanner,
                isPoweredByLinkDisabledByException,
                activeLanguages,
                frontend: { isDataProcessingInUnsafeCountries, isAgeNotice, isListServicesNotice, isTcf, isGcm },
            },
        },
    } = RootStore.get;
    const {
        panel,
        headlines,
        others: {
            customHtmlDecisionButtonTypeDiffers,
            customHtmlDecisionLegalNoticeAll,
            customHtmlDecisionLegalNoticeEssentials,
            customHtmlDecisionLegalNoticeIndividual,
            customHtmlMobileHideHeaderIgnoreDueCloseIcon,
            customHtmlLayoutMaxHeightNotice,
            customHtmlLayoutAnimationInClsNotice,
        },
        sections,
        settings: {
            layout,
            decision,
            design,
            headerDesign,
            bodyDesign,
            footerDesign,
            texts,
            individualLayout,
            group,
            saveButton,
            mobile,
            sticky,
        },
    } = customizeIdsBanner;
    const customize = getSidebarCustomize();

    const isTruthy = (value: string) => !!value;
    const isTruthyAndPro = (value: string) => isTruthy(value) && isPro;
    const isGreaterZero = (value: number) => value > 0;
    const isNotStringHide = (value: string) => value !== "hide";
    const isStringHide = (value: string) => value === "hide";
    const isNotStringNone = (value: string) => value !== "none";
    const isTeachingsEnabled = () => isDataProcessingInUnsafeCountries || isAgeNotice || isListServicesNotice;
    const isTfcDisabled = () => !isTcf;
    const isAccordionNeeded = () => isTcf || isGcm;
    const isListOfServicesDisabled = () => !isListServicesNotice;
    const isMultilingualWebsite = () => activeLanguages.length > 1;

    const hasTooLowMaxHeight = () => {
        const type = customize(layout.type).get();
        const maxHeight = customize(layout.maxHeight).get();
        return (type === "dialog" && maxHeight < 740) || (type === "banner" && maxHeight < 500);
    };

    // Accept essentials inheritance
    const hasDifferentButtonTypesAcceptAllAndEssential = () => {
        return customize(decision.acceptAll).get() !== customize(decision.acceptEssentials).get();
    };
    const hasSameButtonTypesAcceptAllAndEssential = () => !hasDifferentButtonTypesAcceptAllAndEssential();
    const allowAcceptEssentialsUseAcceptAllFn = () => {
        const useAcceptAll = customize(bodyDesign.acceptEssentialsUseAcceptAll).get() as number;
        return useAcceptAll ? hasDifferentButtonTypesAcceptAllAndEssential() : true;
    };
    const showAcceptEssentialsControl = {
        [bodyDesign.acceptEssentialsUseAcceptAll]: allowAcceptEssentialsUseAcceptAllFn,
        [decision.acceptAll]: allowAcceptEssentialsUseAcceptAllFn,
        [decision.acceptEssentials]: allowAcceptEssentialsUseAcceptAllFn,
    };

    // Save button inheritance
    const hasDifferentButtonTypeAcceptAllAndSaveButton = () => {
        return customize(saveButton.type).get() !== customize(decision.acceptAll).get();
    };
    const hasSameButtonTypeAcceptAllAndSaveButton = () => !hasDifferentButtonTypeAcceptAllAndSaveButton();
    const allowSaveButtonUseAcceptAllFn = () => {
        const useAcceptAll = customize(saveButton.useAcceptAll).get() as number;
        return useAcceptAll ? hasDifferentButtonTypeAcceptAllAndSaveButton() : true;
    };
    const showSaveButtonControl = {
        [saveButton.useAcceptAll]: allowSaveButtonUseAcceptAllFn,
        [decision.acceptAll]: allowSaveButtonUseAcceptAllFn,
        [saveButton.type]: allowSaveButtonUseAcceptAllFn,
    };
    const showSaveButtonButtonControl = {
        ...showSaveButtonControl,
        [saveButton.type]: (value: string) => {
            return allowSaveButtonUseAcceptAllFn() && value === "button";
        },
    };

    sidebarFactory({
        panel,
        ids: customizeIdsBanner,
        defaults: customizeDefaultsBanner,
        contentBesideSection: {
            [sections.design]: () => <HeadlineBeforeSection>{__("Cookie banner/dialog design")}</HeadlineBeforeSection>,
            [sections.mobile]: () => <HeadlineBeforeSection>{__("Miscellaneous")}</HeadlineBeforeSection>,
            [sections.individualLayout]: () => (
                <HeadlineBeforeSection>{__("Individual privacy settings")}</HeadlineBeforeSection>
            ),
            [sections.layout]: () => (
                <CustomizeContextProvider>
                    <CustomizePresetSelector />
                    <CustomizeA11yScore />
                </CustomizeContextProvider>
            ),
            [sections.customCss]: {
                render: () => (
                    <>
                        <HeadlineBeforeSection>{__("Useful links")}</HeadlineBeforeSection>
                        <ul style={{ listStyle: "initial", padding: "10px 15px 10px 30px" }}>
                            <li>
                                {_i(
                                    __("Avoid popular mistakes when creating a cookie banner. {{a}}Learn more!{{/a}}"),
                                    {
                                        a: (
                                            <a
                                                href={__("https://devowl.io/cookie-banner/common-mistakes/")}
                                                rel="noreferrer"
                                                target="_blank"
                                            />
                                        ),
                                    },
                                )}
                            </li>
                            <li>
                                {_i(
                                    __(
                                        "Find out what you should pay attention to in the {{a}}wording of your cookie banner{{/a}}!",
                                    ),
                                    {
                                        a: (
                                            <a
                                                href={__("https://devowl.io/cookie-banner/text/")}
                                                rel="noreferrer"
                                                target="_blank"
                                            />
                                        ),
                                    },
                                )}
                            </li>
                            <li>
                                {_i(__("Learn what to look for when {{a}}designing your cookie banner{{/a}}!"), {
                                    a: (
                                        <a
                                            href={__("https://devowl.io/cookie-banner/design/")}
                                            rel="noreferrer"
                                            target="_blank"
                                        />
                                    ),
                                })}
                            </li>
                            <li>
                                {_i(__("Learn more about {{a}}accessibility and the legal conditions!{{/a}}"), {
                                    a: (
                                        <a
                                            href={__(
                                                "https://devowl.io/knowledge-base/accessibility-of-real-cookie-banner/",
                                            )}
                                            rel="noreferrer"
                                            target="_blank"
                                        />
                                    ),
                                })}
                            </li>
                        </ul>
                    </>
                ),
                type: "after",
            },
        },
        controls: {
            acceptAllButtonType: () => <DecisionButtonTypeSelectControl decision="acceptAll" />,
            acceptEssentialsButtonType: () => <DecisionButtonTypeSelectControl decision="acceptEssentials" />,
            acceptIndividualButtonType: () => <DecisionButtonTypeSelectControl decision="acceptIndividual" />,
            rcbGroupsFirstView: () => (
                <CustomizeContextProvider>
                    <GroupsFirstViewControl />
                </CustomizeContextProvider>
            ),
            mobileSwitcher: () => (
                <CustomizeContextProvider>
                    <MobileSwitcherControl />
                </CustomizeContextProvider>
            ),
            stickyLinksSwitcher: () => (
                <CustomizeContextProvider>
                    <StickyLinksControl />
                </CustomizeContextProvider>
            ),
        },
        conditionals: {
            // Basic Layout
            [customHtmlLayoutMaxHeightNotice]: {
                [layout.maxHeightEnabled]: true,
                [layout.maxHeight]: hasTooLowMaxHeight,
                [layout.type]: hasTooLowMaxHeight,
            },
            [layout.maxHeight]: {
                [layout.maxHeightEnabled]: true,
            },
            [layout.dialogMaxWidth]: {
                [layout.type]: "dialog",
            },
            [layout.dialogMargin]: {
                [layout.type]: "dialog",
                [layout.dialogPosition]: (value: Customize["layout"]["dialogPosition"]) => value !== "middleCenter",
            },
            [layout.dialogPosition]: {
                [layout.type]: "dialog",
            },
            [layout.bannerPosition]: {
                [layout.type]: "banner",
            },
            [layout.bannerMaxWidth]: {
                [layout.type]: "banner",
            },
            [layout.animationInDuration]: {
                [layout.animationIn]: isNotStringNone,
            },
            [layout.animationInOnlyMobile]: {
                [layout.animationIn]: isNotStringNone,
            },
            [layout.animationOutDuration]: {
                [layout.animationOut]: isNotStringNone,
            },
            [layout.animationOutOnlyMobile]: {
                [layout.animationOut]: isNotStringNone,
            },
            [layout.dialogBorderRadius]: {
                [layout.type]: "dialog",
            },
            [layout.overlayBg]: {
                [layout.overlay]: true,
            },
            [layout.overlayBgAlpha]: {
                [layout.overlay]: true,
            },
            [layout.overlayBlur]: {
                [layout.overlay]: true,
            },
            [customHtmlLayoutAnimationInClsNotice]: {
                [layout.animationIn]: (value: string) =>
                    ["none", "fadeIn"]
                        .concat(
                            process.env.PLUGIN_CTX === "pro"
                                ? [
                                      "bounce",
                                      "bounceIn",
                                      "flash",
                                      "flipInX",
                                      "flipInY",
                                      "headShake",
                                      "jackInTheBox",
                                      "jello",
                                      "lightSpeedIn",
                                      "pulse",
                                      "rollIn",
                                      "rotateIn",
                                      "rotateInDownLeft",
                                      "rotateInDownRight",
                                      "rotateInUpLeft",
                                      "rotateInUpRight",
                                      "rubberBand",
                                      "shake",
                                      "swing",
                                      "tada",
                                      "wobble",
                                      "zoomIn",
                                      "zoomInDown",
                                      "zoomInLeft",
                                      "zoomInRight",
                                      "zoomInUp",
                                  ]
                                : [],
                        )
                        .indexOf(value) > -1,
            },
            // Content decision
            [customHtmlDecisionLegalNoticeAll]: {
                [decision.acceptAll]: isStringHide,
            },
            // Content decision
            [customHtmlDecisionButtonTypeDiffers]: {
                [decision.acceptAll]: hasDifferentButtonTypesAcceptAllAndEssential,
                [decision.acceptEssentials]: hasDifferentButtonTypesAcceptAllAndEssential,
                ...showAcceptEssentialsControl,
            },
            [customHtmlDecisionLegalNoticeEssentials]: {
                [decision.acceptEssentials]: isStringHide,
                ...showAcceptEssentialsControl,
            },
            [customHtmlDecisionLegalNoticeIndividual]: {
                [decision.acceptIndividual]: isStringHide,
            },
            [decision.groupsFirstView]: {
                [decision.groupsFirstView]: isTfcDisabled,
                [decision.showGroups]: true,
            },
            [decision.saveButton]: {
                [decision.groupsFirstView]: true,
                [decision.showGroups]: true,
            },
            // Design
            [design.borderColor]: {
                [design.borderWidth]: isGreaterZero,
            },
            [design.fontFamily]: {
                [design.fontInheritFamily]: false,
            },
            [design.boxShadowOffsetX]: {
                [design.boxShadowEnabled]: true,
            },
            [design.boxShadowOffsetY]: {
                [design.boxShadowEnabled]: true,
            },
            [design.boxShadowBlurRadius]: {
                [design.boxShadowEnabled]: true,
            },
            [design.boxShadowSpreadRadius]: {
                [design.boxShadowEnabled]: true,
            },
            [design.boxShadowColor]: {
                [design.boxShadowEnabled]: true,
            },
            [design.boxShadowColorAlpha]: {
                [design.boxShadowEnabled]: true,
            },
            // Header design
            [headerDesign.bg]: {
                [headerDesign.inheritBg]: false,
            },
            [headerDesign.textAlign]: {
                [headerDesign.inheritTextAlign]: false,
            },
            [headerDesign.logoRetina]: {
                [headerDesign.logo]: (value: string) => value && !value.endsWith(".svg"),
            },
            [headerDesign.logoMaxHeight]: {
                [headerDesign.logo]: isTruthy,
            },
            [headerDesign.logoPosition]: {
                [headerDesign.logo]: isTruthy,
                [texts.headline]: isTruthy,
            },
            [headerDesign.logoMargin]: {
                [headerDesign.logo]: isTruthy,
            },
            [headerDesign.fontFamily]: {
                [headerDesign.fontInheritFamily]: false,
            },
            [headerDesign.fontSize]: {
                [texts.headline]: isTruthy,
            },
            [headerDesign.fontFamily]: {
                [headerDesign.fontInheritFamily]: false,
            },
            [headerDesign.borderColor]: {
                [headerDesign.borderWidth]: isGreaterZero,
            },
            // Body design
            [bodyDesign.descriptionFontSize]: {
                [bodyDesign.descriptionInheritFontSize]: false,
            },
            [bodyDesign.dottedGroupsInheritFontSize]: {
                [bodyDesign.dottedGroupsInheritFontSize]: isTfcDisabled,
            },
            [bodyDesign.dottedGroupsFontSize]: {
                [bodyDesign.dottedGroupsInheritFontSize]: false,
                [bodyDesign.dottedGroupsFontSize]: isTfcDisabled,
            },
            [bodyDesign.dottedGroupsBulletColor]: {
                [bodyDesign.dottedGroupsBulletColor]: isTfcDisabled,
                [decision.showGroups]: true,
            },
            // Additional teachings
            [headlines.bodyDesignTeachings]: {
                [headlines.bodyDesignTeachings]: isTeachingsEnabled,
            },
            [headlines.bodyDesignTeachingsFont]: {
                [headlines.bodyDesignTeachingsFont]: isTeachingsEnabled,
            },
            [headlines.bodyDesignTeachingsSeparator]: {
                [bodyDesign.teachingsSeparatorActive]: true,
                [headlines.bodyDesignTeachingsSeparator]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsInheritTextAlign]: {
                [bodyDesign.teachingsInheritTextAlign]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsTextAlign]: {
                [bodyDesign.teachingsInheritTextAlign]: false,
                [bodyDesign.teachingsTextAlign]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsInheritFontSize]: {
                [bodyDesign.teachingsInheritFontSize]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsFontSize]: {
                [bodyDesign.teachingsInheritFontSize]: false,
                [bodyDesign.teachingsFontSize]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsInheritTextAlign]: {
                [bodyDesign.teachingsInheritTextAlign]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsFontColor]: {
                [bodyDesign.teachingsInheritFontColor]: false,
                [bodyDesign.teachingsFontColor]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsSeparatorWidth]: {
                [bodyDesign.teachingsSeparatorActive]: true,
                [bodyDesign.teachingsSeparatorWidth]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsSeparatorHeight]: {
                [bodyDesign.teachingsSeparatorActive]: true,
                [bodyDesign.teachingsSeparatorHeight]: isTeachingsEnabled,
            },
            [bodyDesign.teachingsSeparatorColor]: {
                [bodyDesign.teachingsSeparatorActive]: true,
                [bodyDesign.teachingsSeparatorColor]: isTeachingsEnabled,
            },
            // TCF Stacks
            ...(process.env.PLUGIN_CTX === "pro"
                ? {
                      [headlines.bodyDesignAccordion]: {
                          [headlines.bodyDesignAccordion]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionMargin]: {
                          [bodyDesign.accordionMargin]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionPadding]: {
                          [bodyDesign.accordionPadding]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionArrowType]: {
                          [bodyDesign.accordionArrowType]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionArrowColor]: {
                          [bodyDesign.accordionArrowColor]: isAccordionNeeded,
                          [bodyDesign.accordionArrowType]: isNotStringNone,
                      },
                      [bodyDesign.accordionBg]: {
                          [bodyDesign.accordionBg]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionActiveBg]: {
                          [bodyDesign.accordionActiveBg]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionHoverBg]: {
                          [bodyDesign.accordionHoverBg]: isAccordionNeeded,
                      },
                      [headlines.bodyDesignAccordionBorder]: {
                          [headlines.bodyDesignAccordionBorder]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionBorderWidth]: {
                          [bodyDesign.accordionBorderWidth]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionBorderColor]: {
                          [bodyDesign.accordionBorderColor]: isAccordionNeeded,
                          [bodyDesign.accordionBorderWidth]: isGreaterZero,
                      },
                      [headlines.bodyDesignAccordionTitle]: {
                          [headlines.bodyDesignAccordionTitle]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionTitleFontSize]: {
                          [bodyDesign.accordionTitleFontSize]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionTitleFontColor]: {
                          [bodyDesign.accordionTitleFontColor]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionTitleFontWeight]: {
                          [bodyDesign.accordionTitleFontWeight]: isAccordionNeeded,
                      },
                      [headlines.bodyDesignAccordionDescription]: {
                          [headlines.bodyDesignAccordionDescription]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionDescriptionMargin]: {
                          [bodyDesign.accordionDescriptionMargin]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionDescriptionFontSize]: {
                          [bodyDesign.accordionDescriptionFontSize]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionDescriptionFontColor]: {
                          [bodyDesign.accordionDescriptionFontColor]: isAccordionNeeded,
                      },
                      [bodyDesign.accordionDescriptionFontWeight]: {
                          [bodyDesign.accordionDescriptionFontWeight]: isAccordionNeeded,
                      },
                      [headlines.textsTcfStacks]: {
                          [headlines.textsTcfStacks]: isListOfServicesDisabled,
                      },
                      [texts.tcfStacksCustomName]: {
                          [texts.tcfStacksCustomName]: isListOfServicesDisabled,
                      },
                      [texts.tcfStacksCustomDescription]: {
                          [texts.tcfStacksCustomDescription]: isListOfServicesDisabled,
                      },
                  }
                : {}),
            // Button: Accept all
            [bodyDesign.acceptAllOneRowLayout]: {
                [decision.acceptAll]: isNotStringHide,
                [decision.acceptEssentials]: isNotStringHide,
            },
            [bodyDesign.acceptAllFontSize]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllPadding]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllBg]: {
                [decision.acceptAll]: "button",
            },
            [bodyDesign.acceptAllTextAlign]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllBorderWidth]: {
                [decision.acceptAll]: "button",
            },
            [bodyDesign.acceptAllBorderColor]: {
                [decision.acceptAll]: "button",
                [bodyDesign.acceptAllBorderWidth]: isGreaterZero,
            },
            [bodyDesign.acceptAllFontColor]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllFontWeight]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllHoverBg]: {
                [decision.acceptAll]: "button",
            },
            [bodyDesign.acceptAllHoverFontColor]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [bodyDesign.acceptAllHoverBorderColor]: {
                [decision.acceptAll]: "button",
                [bodyDesign.acceptAllBorderWidth]: isGreaterZero,
            },
            [headlines.bodyDesignAcceptAllBorder]: {
                [decision.acceptAll]: "button",
            },
            [headlines.bodyDesignAcceptAllFont]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [headlines.bodyDesignAcceptAllHover]: {
                [decision.acceptAll]: isNotStringHide,
            },
            // Button: Continue without consent
            [bodyDesign.acceptEssentialsUseAcceptAll]: {
                [decision.acceptAll]: hasSameButtonTypesAcceptAllAndEssential,
                [decision.acceptEssentials]: hasSameButtonTypesAcceptAllAndEssential,
            },
            [bodyDesign.acceptEssentialsFontSize]: {
                [decision.acceptEssentials]: isNotStringHide,
                [bodyDesign.acceptAllOneRowLayout]: false,
                [bodyDesign.acceptEssentialsFontSize]: isTfcDisabled,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsPadding]: {
                [decision.acceptEssentials]: isNotStringHide,
                [bodyDesign.acceptAllOneRowLayout]: false,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsBg]: {
                [decision.acceptEssentials]: "button",
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsTextAlign]: {
                [decision.acceptEssentials]: isNotStringHide,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsBorderWidth]: {
                [decision.acceptEssentials]: "button",
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsBorderColor]: {
                [decision.acceptEssentials]: "button",
                [bodyDesign.acceptEssentialsBorderWidth]: isGreaterZero,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsFontColor]: {
                [decision.acceptEssentials]: isNotStringHide,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsFontWeight]: {
                [decision.acceptEssentials]: isNotStringHide,
                [bodyDesign.acceptEssentialsFontWeight]: isTfcDisabled,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsHoverBg]: {
                [decision.acceptEssentials]: "button",
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsHoverFontColor]: {
                [decision.acceptEssentials]: isNotStringHide,
                ...showAcceptEssentialsControl,
            },
            [bodyDesign.acceptEssentialsHoverBorderColor]: {
                [decision.acceptEssentials]: "button",
                [bodyDesign.acceptEssentialsBorderWidth]: isGreaterZero,
                ...showAcceptEssentialsControl,
            },
            [headlines.bodyDesignAcceptEssentialsBorder]: {
                [decision.acceptEssentials]: "button",
                ...showAcceptEssentialsControl,
            },
            [headlines.bodyDesignAcceptEssentialsFont]: {
                [decision.acceptEssentials]: isNotStringHide,
                ...showAcceptEssentialsControl,
            },
            [headlines.bodyDesignAcceptEssentialsHover]: {
                [decision.acceptEssentials]: isNotStringHide,
                ...showAcceptEssentialsControl,
            },
            // Button: Individual privacy preferences
            [bodyDesign.acceptIndividualFontSize]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualPadding]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualBg]: {
                [decision.acceptIndividual]: "button",
            },
            [bodyDesign.acceptIndividualTextAlign]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualBorderWidth]: {
                [decision.acceptIndividual]: "button",
            },
            [bodyDesign.acceptIndividualBorderColor]: {
                [decision.acceptIndividual]: "button",
                [bodyDesign.acceptIndividualBorderWidth]: isGreaterZero,
            },
            [bodyDesign.acceptIndividualFontColor]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualFontWeight]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualHoverBg]: {
                [decision.acceptIndividual]: "button",
            },
            [bodyDesign.acceptIndividualHoverFontColor]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [bodyDesign.acceptIndividualHoverBorderColor]: {
                [decision.acceptIndividual]: "button",
                [bodyDesign.acceptIndividualBorderWidth]: isGreaterZero,
            },
            [headlines.bodyDesignAcceptIndividualBorder]: {
                [decision.acceptIndividual]: "button",
            },
            [headlines.bodyDesignAcceptIndividualFont]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [headlines.bodyDesignAcceptIndividualHover]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            // Footer design
            [footerDesign.poweredByLink]: {
                [footerDesign.poweredByLink]: () => !isPoweredByLinkDisabledByException,
            },
            [footerDesign.bg]: {
                [footerDesign.inheritBg]: false,
            },
            [footerDesign.textAlign]: {
                [footerDesign.inheritTextAlign]: false,
            },
            [footerDesign.fontFamily]: {
                [design.fontInheritFamily]: false,
            },
            [footerDesign.fontSize]: {
                [texts.headline]: isTruthy,
            },
            [footerDesign.fontFamily]: {
                [footerDesign.fontInheritFamily]: false,
            },
            [footerDesign.borderColor]: {
                [footerDesign.borderWidth]: isGreaterZero,
            },
            [headlines.footerLanguageSwitcher]: {
                [headlines.footerLanguageSwitcher]: isMultilingualWebsite,
            },
            [footerDesign.languageSwitcher]: {
                [footerDesign.languageSwitcher]: isMultilingualWebsite,
            },
            // Texts
            [texts.acceptAll]: {
                [decision.acceptAll]: isNotStringHide,
            },
            [texts.acceptEssentials]: {
                [decision.acceptEssentials]: isNotStringHide,
            },
            [texts.acceptIndividual]: {
                [decision.acceptIndividual]: isNotStringHide,
            },
            [texts.poweredBy]: {
                [footerDesign.poweredByLink]: true,
            },
            // Individual layout
            [individualLayout.inheritDialogMaxWidth]: {
                [layout.type]: "dialog",
            },
            [individualLayout.dialogMaxWidth]: {
                [layout.type]: "dialog",
                [individualLayout.inheritDialogMaxWidth]: false,
            },
            [individualLayout.inheritBannerMaxWidth]: {
                [layout.type]: "banner",
            },
            [individualLayout.bannerMaxWidth]: {
                [layout.type]: "banner",
                [individualLayout.inheritBannerMaxWidth]: false,
            },
            // Cookie groups
            [group.checkboxBorderColor]: {
                [group.checkboxBorderWidth]: isGreaterZero,
            },
            [group.checkboxActiveBorderColor]: {
                [group.checkboxBorderWidth]: isGreaterZero,
            },
            [group.groupBg]: {
                [group.groupInheritBg]: false,
            },
            [group.groupBorderColor]: {
                [group.groupBorderWidth]: isGreaterZero,
            },
            // Save button
            [saveButton.useAcceptAll]: {
                [decision.acceptAll]: hasSameButtonTypeAcceptAllAndSaveButton,
                [saveButton.type]: hasSameButtonTypeAcceptAllAndSaveButton,
            },
            [saveButton.type]: showSaveButtonControl,
            [saveButton.padding]: showSaveButtonControl,
            [saveButton.bg]: showSaveButtonButtonControl,
            [saveButton.textAlign]: showSaveButtonControl,
            [headlines.saveButtonFont]: showSaveButtonControl,
            [saveButton.fontColor]: showSaveButtonControl,
            [saveButton.fontSize]: {
                [saveButton.fontSize]: isTfcDisabled,
                ...showSaveButtonControl,
            },
            [saveButton.fontWeight]: {
                [saveButton.fontWeight]: isTfcDisabled,
                ...showSaveButtonControl,
            },
            [saveButton.borderWidth]: showSaveButtonButtonControl,
            [saveButton.borderColor]: {
                [saveButton.borderWidth]: isGreaterZero,
                ...showSaveButtonButtonControl,
            },
            [headlines.saveButtonHover]: showSaveButtonControl,
            [saveButton.hoverBg]: showSaveButtonButtonControl,
            [saveButton.hoverFontColor]: showSaveButtonControl,
            [saveButton.hoverBorderColor]: {
                [saveButton.borderWidth]: isGreaterZero,
                ...showSaveButtonButtonControl,
            },
            [headlines.saveButtonBorder]: showSaveButtonButtonControl,
            // Mobile experience
            [mobile.hideHeader]: {
                [mobile.enabled]: isTruthyAndPro,
            },
            [mobile.maxHeight]: {
                [mobile.enabled]: isTruthyAndPro,
            },
            [mobile.alignment]: {
                [mobile.enabled]: isTruthyAndPro,
            },
            [mobile.scalePercent]: {
                [mobile.enabled]: isTruthyAndPro,
            },
            [mobile.scalePercentVertical]: {
                [mobile.enabled]: isTruthyAndPro,
            },
            [customHtmlMobileHideHeaderIgnoreDueCloseIcon]: {
                [mobile.hideHeader]: true,
                [decision.showCloseIcon]: true,
                [decision.acceptEssentials]: isStringHide,
            },
            // Sticky legal links widget
            [sticky.bubbleBorderColor]: {
                [sticky.bubbleBorderWidth]: isGreaterZero,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleHoverBorderColor]: {
                [sticky.bubbleBorderWidth]: isGreaterZero,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowOffsetX]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowOffsetY]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowBlurRadius]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowSpreadRadius]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowColor]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowColorAlpha]: {
                [sticky.boxShadowEnabled]: true,
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.animationsEnabled]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.alignment]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleBorderRadius]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.icon]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.iconCustom]: {
                [sticky.enabled]: isTruthyAndPro,
                [sticky.icon]: (value) => value === "custom",
            },
            [sticky.iconCustomRetina]: {
                [sticky.enabled]: isTruthyAndPro,
                [sticky.iconCustom]: (value: string) => value && !value.endsWith(".svg"),
                [sticky.icon]: (value) => value === "custom",
            },
            [sticky.iconSize]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.iconColor]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleMargin]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubblePadding]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleBg]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleBorderWidth]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.boxShadowEnabled]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleHoverBg]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.bubbleHoverBorderColor]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.hoverIconColor]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.hoverIconCustom]: {
                [sticky.enabled]: isTruthyAndPro,
                [sticky.icon]: (value) => value === "custom",
            },
            [sticky.hoverIconCustomRetina]: {
                [sticky.enabled]: isTruthyAndPro,
                [sticky.hoverIconCustom]: (value: string) => value && !value.endsWith(".svg"),
                [sticky.icon]: (value) => value === "custom",
            },
            [sticky.menuFontSize]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.menuBorderRadius]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.menuItemSpacing]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [sticky.menuItemPadding]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.stickyIcon]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.stickyBorder]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.stickyBoxShadow]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.stickyHover]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.stickyMenu]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [headlines.textsSticky]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [texts.stickyChange]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [texts.stickyHistory]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [texts.stickyRevoke]: {
                [sticky.enabled]: isTruthyAndPro,
            },
            [texts.stickyRevokeSuccessMessage]: {
                [sticky.enabled]: isTruthyAndPro,
            },
        },
        onReady: () => {
            // Add Pro tags to some controls
            applyCustomizeProTag(layout.overlayBlur, {
                title: __("Want to stand out from the crowd?"),
                testDrive: true,
                feature: "blur",
                assetName: __("pro-modal/blur.png"),
                description: __(
                    "Blur the background and set the focus on the cookie banner so that your visitors agree to as many services and cookies as possible.",
                ),
            });
            applyCustomizeProTag(layout.animationIn, {
                title: __("Want to have cool animations?"),
                testDrive: true,
                feature: "animations-in",
                assetName: __("pro-modal/animations-in.png"),
                description: __(
                    "The cookie banner can be displayed with an animation. Choose from over 40 animations the most suitable!",
                ),
                tagText: __("Unlock animations"),
            });
            applyCustomizeProTag(layout.animationOut, {
                title: __("Want to have cool animations?"),
                testDrive: true,
                feature: "animations-out",
                assetName: __("pro-modal/animations-in.png"),
                description: __(
                    "The cookie banner can be hidden with an animation. Choose from over 40 animations the most suitable!",
                ),
                tagText: __("Unlock animations"),
            });

            const buttonConstrastRatio = (
                settings: Pick<Parameters<typeof a11yConstrastRatio>[0], "settings" | "modifier">,
            ) =>
                a11yConstrastRatio({
                    ...settings,
                    minimumRatio: isTcf ? MIN_TCF_RATIO : MIN_RATIO,
                    severity: isTcf ? "error" : "warning",
                    renderNotice: isTcf
                        ? () =>
                              __(
                                  "You need to use a minimum contrast ratio between background and font color of %d to be compliant with TCF!",
                                  MIN_TCF_RATIO,
                              )
                        : undefined,
                });

            const miscContrastRatio = (
                settings: Pick<Parameters<typeof a11yConstrastRatio>[0], "settings" | "modifier">,
            ) =>
                a11yConstrastRatio({
                    ...settings,
                    minimumRatio: MIN_RATIO,
                    severity: "warning",
                });

            miscContrastRatio({
                settings: [design.bg, design.fontColor],
            });

            miscContrastRatio({
                settings: [headerDesign.bg, headerDesign.fontColor, headerDesign.inheritBg, design.bg],
                modifier: ([bg, fontColor], [inheritBg, designBg]) => [inheritBg ? designBg : bg, fontColor],
            });

            miscContrastRatio({
                settings: [design.bg, bodyDesign.teachingsFontColor],
            });

            miscContrastRatio({
                settings: [footerDesign.bg, footerDesign.fontColor, footerDesign.inheritBg, design.bg],
                modifier: ([bg, fontColor], [inheritBg, designBg]) => [inheritBg ? designBg : bg, fontColor],
            });

            miscContrastRatio({
                settings: [footerDesign.bg, footerDesign.hoverFontColor, footerDesign.inheritBg, design.bg],
                modifier: ([bg, hoverFontColor], [inheritBg, designBg]) => [inheritBg ? designBg : bg, hoverFontColor],
            });

            miscContrastRatio({
                settings: [group.checkboxActiveBg, group.checkboxActiveColor],
            });

            miscContrastRatio({
                settings: [group.groupBg, group.headlineFontColor, group.groupInheritBg, design.bg],
                modifier: ([groupBg, headlineFontColor], [groupInheritBg, designBg]) => [
                    groupInheritBg ? designBg : groupBg,
                    headlineFontColor,
                ],
            });

            miscContrastRatio({
                settings: [group.groupBg, group.descriptionFontColor, group.groupInheritBg, design.bg],
                modifier: ([groupBg, descriptionFontColor], [groupInheritBg, designBg]) => [
                    groupInheritBg ? designBg : groupBg,
                    descriptionFontColor,
                ],
            });

            miscContrastRatio({
                settings: [group.groupBg, group.linkColor, group.groupInheritBg, design.bg],
                modifier: ([groupBg, linkColor], [groupInheritBg, designBg]) => [
                    groupInheritBg ? designBg : groupBg,
                    linkColor,
                ],
            });

            miscContrastRatio({
                settings: [group.groupBg, group.linkHoverColor, group.groupInheritBg, design.bg],
                modifier: ([groupBg, linkHoverColor], [groupInheritBg, designBg]) => [
                    groupInheritBg ? designBg : groupBg,
                    linkHoverColor,
                ],
            });

            buttonConstrastRatio({
                settings: [bodyDesign.acceptAllBg, bodyDesign.acceptAllFontColor, decision.acceptAll, design.bg],
                modifier: ([acceptAllBg, acceptAllFontColor], [decisionAcceptAll, designBg]) => [
                    // If button is "link", use the main background color
                    (decisionAcceptAll as Customize["decision"]["acceptAll"]) === "link" ? designBg : acceptAllBg,
                    acceptAllFontColor,
                ],
            });
            buttonConstrastRatio({
                settings: [
                    bodyDesign.acceptAllHoverBg,
                    bodyDesign.acceptAllHoverFontColor,
                    decision.acceptAll,
                    design.bg,
                ],
                modifier: ([acceptAllHoverBg, acceptAllHoverFontColor], [decisionAcceptAll, designBg]) => [
                    // If button is "link", use the main background color
                    (decisionAcceptAll as Customize["decision"]["acceptAll"]) === "link" ? designBg : acceptAllHoverBg,
                    acceptAllHoverFontColor,
                ],
            });

            buttonConstrastRatio({
                settings: [
                    bodyDesign.acceptEssentialsBg,
                    bodyDesign.acceptEssentialsFontColor,
                    decision.acceptEssentials,
                    design.bg,
                ],
                modifier: ([acceptEssentialsBg, acceptEssentialsFontColor], [decisionAcceptEssentials, designBg]) => [
                    // If button is "link", use the main background color
                    (decisionAcceptEssentials as Customize["decision"]["acceptEssentials"]) === "link"
                        ? designBg
                        : acceptEssentialsBg,
                    acceptEssentialsFontColor,
                ],
            });
            buttonConstrastRatio({
                settings: [
                    bodyDesign.acceptEssentialsHoverBg,
                    bodyDesign.acceptEssentialsHoverFontColor,
                    decision.acceptEssentials,
                    design.bg,
                ],
                modifier: (
                    [acceptEssentialsHoverBg, acceptEssentialsHoverFontColor],
                    [decisionAcceptEssentials, designBg],
                ) => [
                    // If button is "link", use the main background color
                    (decisionAcceptEssentials as Customize["decision"]["acceptEssentials"]) === "link"
                        ? designBg
                        : acceptEssentialsHoverBg,
                    acceptEssentialsHoverFontColor,
                ],
            });

            buttonConstrastRatio({
                settings: [
                    saveButton.bg,
                    saveButton.fontColor,
                    saveButton.type,
                    saveButton.useAcceptAll,
                    decision.acceptAll,
                    design.bg,
                ],
                modifier: (
                    [saveButtonBg, saveButtonFontColor],
                    [saveButtonType, saveButtonUseAcceptAll, decisionAcceptAll, designBg],
                ) => {
                    return [
                        // If button is "link", use the main background color
                        (saveButtonUseAcceptAll
                            ? (decisionAcceptAll as Customize["decision"]["acceptAll"])
                            : (saveButtonType as Customize["saveButton"]["type"])) === "link"
                            ? designBg
                            : saveButtonBg,
                        saveButtonFontColor,
                    ];
                },
            });
            buttonConstrastRatio({
                settings: [
                    saveButton.hoverBg,
                    saveButton.hoverFontColor,
                    saveButton.type,
                    saveButton.useAcceptAll,
                    decision.acceptAll,
                    design.bg,
                ],
                modifier: (
                    [saveButtonHoverBg, saveButtonHoverFontColor],
                    [saveButtonType, saveButtonUseAcceptAll, decisionAcceptAll, designBg],
                ) => {
                    return [
                        // If button is "link", use the main background color
                        (saveButtonUseAcceptAll
                            ? (decisionAcceptAll as Customize["decision"]["acceptAll"])
                            : (saveButtonType as Customize["saveButton"]["type"])) === "link"
                            ? designBg
                            : saveButtonHoverBg,
                        saveButtonHoverFontColor,
                    ];
                },
            });

            buttonConstrastRatio({
                settings: [
                    bodyDesign.acceptIndividualBg,
                    bodyDesign.acceptIndividualFontColor,
                    decision.acceptIndividual,
                    design.bg,
                ],
                modifier: ([acceptIndividualBg, acceptIndividualFontColor], [decisionIndividual, designBg]) => [
                    // If button is "link", use the main background color
                    (decisionIndividual as Customize["decision"]["acceptIndividual"]) === "link"
                        ? designBg
                        : acceptIndividualBg,
                    acceptIndividualFontColor,
                ],
            });
            buttonConstrastRatio({
                settings: [
                    bodyDesign.acceptIndividualHoverBg,
                    bodyDesign.acceptIndividualHoverFontColor,
                    decision.acceptIndividual,
                    design.bg,
                ],
                modifier: (
                    [acceptIndividualHoverBg, acceptIndividualHoverFontColor],
                    [decisionIndividual, designBg],
                ) => [
                    // If button is "link", use the main background color
                    (decisionIndividual as Customize["decision"]["acceptIndividual"]) === "link"
                        ? designBg
                        : acceptIndividualHoverBg,
                    acceptIndividualHoverFontColor,
                ],
            });

            const miscFontWeight = (settings: Pick<Parameters<typeof a11yFontWeight>[0], "settings" | "modifier">) =>
                a11yFontWeight({
                    ...settings,
                    notAccessible: ["lighter"],
                    severity: "warning",
                });

            miscFontWeight({
                settings: [design.fontWeight],
            });
            miscFontWeight({
                settings: [headerDesign.fontWeight],
            });
            miscFontWeight({
                settings: [bodyDesign.acceptAllFontWeight],
            });
            miscFontWeight({
                settings: [bodyDesign.acceptEssentialsFontWeight],
            });
            miscFontWeight({
                settings: [bodyDesign.acceptIndividualFontWeight],
            });
            miscFontWeight({
                settings: [footerDesign.fontWeight],
            });
            miscFontWeight({
                settings: [group.headlineFontWeight],
            });
            miscFontWeight({
                settings: [saveButton.fontWeight],
            });

            const availableFontSizes: string[] = [];
            const miscFontSize = (
                settings: Pick<Parameters<typeof a11yFontSize>[0], "settings" | "modifier" | "minimumPx">,
            ) => {
                availableFontSizes.push(settings.settings[0]);
                a11yFontSize({
                    ...settings,
                    severity: "warning",
                });
            };

            miscFontSize({
                settings: [design.fontSize],
                minimumPx: 12,
            });
            miscFontSize({
                settings: [headerDesign.fontSize],
                minimumPx: 16,
            });
            miscFontSize({
                settings: [bodyDesign.descriptionFontSize],
                minimumPx: 12,
            });
            miscFontSize({
                settings: [bodyDesign.dottedGroupsFontSize],
                minimumPx: 12,
            });
            miscFontSize({
                settings: [bodyDesign.teachingsFontSize],
                minimumPx: 12,
            });
            miscFontSize({
                settings: [bodyDesign.acceptAllFontSize],
                minimumPx: 16,
            });
            miscFontSize({
                settings: [bodyDesign.acceptEssentialsFontSize],
                minimumPx: 16,
            });
            miscFontSize({
                settings: [bodyDesign.acceptIndividualFontSize],
                minimumPx: 16,
            });
            miscFontSize({
                settings: [group.headlineFontSize],
                minimumPx: 14,
            });
            miscFontSize({
                settings: [group.descriptionFontSize],
                minimumPx: 12,
            });
            miscFontSize({
                settings: [saveButton.fontSize],
                minimumPx: 16,
            });

            a11yTextDecoration({
                settings: [design.linkTextDecoration],
                notAccessible: ["none"],
                severity: "warning",
            });

            if (process.env.PLUGIN_CTX === "pro") {
                /* onlypro:start */
                miscFontSize({
                    settings: [bodyDesign.accordionTitleFontSize],
                    minimumPx: 12,
                });
                miscFontSize({
                    settings: [bodyDesign.accordionTitleFontSize],
                    minimumPx: 12,
                });
                miscFontWeight({
                    settings: [bodyDesign.accordionTitleFontWeight],
                });
                miscFontWeight({
                    settings: [bodyDesign.accordionDescriptionFontWeight],
                });
                miscContrastRatio({
                    settings: [bodyDesign.accordionBg, bodyDesign.accordionTitleFontColor],
                });
                miscContrastRatio({
                    settings: [bodyDesign.accordionBg, bodyDesign.accordionDescriptionFontColor],
                });

                miscFontSize({
                    settings: [sticky.menuFontSize],
                    minimumPx: 12,
                });
                miscFontSize({
                    settings: [sticky.iconSize],
                    minimumPx: 24,
                });
                miscContrastRatio({
                    settings: [sticky.bubbleBg, sticky.iconColor],
                });
                miscContrastRatio({
                    settings: [sticky.bubbleHoverBg, sticky.hoverIconColor],
                });

                // Custom accessibility-check: Mobile experience
                const settings = [mobile.scalePercent, ...availableFontSizes];
                getSidebarCustomize().control(settings[0], ({ container }: any) => {
                    createRoot(
                        jQuery("<div />")
                            .addClass("customize-control-a11y-mobile-experience")
                            .appendTo(container)
                            .get(0),
                    ).render(<CustomizeA11yMobileExperience settings={settings} />);
                });
                /* onlypro:end */
            }

            // Custom accessibility-check: Overlay
            const settings = [layout.type, layout.overlay, layout.overlayBgAlpha];
            getSidebarCustomize().control(settings[1], ({ container }: any) => {
                createRoot(
                    jQuery("<div />").addClass("customize-control-a11y-overlay").appendTo(container).get(0),
                ).render(<CustomizeA11yOverlay settings={settings} />);
            });
        },
    });
}

export { applyCustomizeBannerSidebar };
