import { observer } from "mobx-react";
import React, { useCallback, useEffect } from "react";

import {
    Avatar,
    Card,
    Form,
    PluginUpdateEmbed,
    Space,
    useStores as useRpmStores,
} from "@devowl-wp/real-product-manager-wp-client";

import { i18n, rmlOpts } from "../../util/index.js";

const ConfigLicensingConsumer = observer(({ withinModal = false, onClose }) => {
    const {
        pluginUpdateStore: { pluginUpdates },
    } = useRpmStores();
    const {
        slug,
        others: { isPro, showLicenseFormImmediate },
        publicUrl,
    } = rmlOpts;
    const pluginUpdate = pluginUpdates.get(slug);

    const handleLicenseUpdated = useCallback(() => {
        // When form is shown directly after first usage, return to dashboard
        rmlOpts.others.showLicenseFormImmediate = false;
        onClose?.();
    }, []);

    const handleLicenseSkip = useCallback(() => {
        rmlOpts.others.showLicenseFormImmediate = false;
        pluginUpdate.skip();
        onClose?.();
    }, [pluginUpdate]);

    // Automatically skip if form already interacted (e.g. browser back cache issue)
    useEffect(() => {
        if (showLicenseFormImmediate && pluginUpdate?.hasInteractedWithFormOnce) {
            handleLicenseSkip();
        }
    }, [pluginUpdate, showLicenseFormImmediate, handleLicenseSkip]);

    return (
        <>
            {showLicenseFormImmediate && (
                <div style={{ maxWidth: 650, textAlign: "center", margin: "0 auto" }}>
                    {!withinModal && (
                        <Avatar
                            src={`${publicUrl}images/logos/real-media-library.svg`}
                            shape="square"
                            size={130}
                            style={{ backgroundColor: "white", padding: 25, borderRadius: 999, marginBottom: 10 }}
                        />
                    )}
                    {!withinModal && !isPro && (
                        <p style={{ fontSize: 15, margin: "0 auto 20px" }}>{i18n("licenseFreeDescription")}</p>
                    )}
                </div>
            )}
            <Card
                title={
                    withinModal || !pluginUpdate
                        ? undefined
                        : isPro || pluginUpdate.isLicensed
                          ? i18n("licenseActivation")
                          : i18n("licenseGetFreeActivation")
                }
                bordered={!withinModal}
            >
                <Space direction="vertical" size="large">
                    {withinModal && !isPro && (
                        <p className="description" style={{ fontWeight: "bold" }}>
                            {i18n("licenseFreeDescription")}
                        </p>
                    )}
                    <PluginUpdateEmbed
                        formProps={{
                            onSave: handleLicenseUpdated,
                            // Skip remote errors and move to dashboard immediately
                            onFailure: showLicenseFormImmediate && !isPro ? handleLicenseSkip : undefined,
                            footer: (
                                <Form.Item
                                    style={{
                                        margin: "25px 0 0",
                                        textAlign: showLicenseFormImmediate ? "center" : undefined,
                                    }}
                                >
                                    <button type="submit" className="button button-primary">
                                        {showLicenseFormImmediate
                                            ? isPro
                                                ? i18n("licenseActivationBtnPro")
                                                : i18n("licenseActivationBtnFree")
                                            : i18n("save")}
                                    </button>
                                </Form.Item>
                            ),
                        }}
                        listProps={{
                            onDeactivate: handleLicenseUpdated,
                        }}
                        slug={slug}
                    />
                </Space>
            </Card>
            {pluginUpdate && showLicenseFormImmediate && (
                <div style={{ textAlign: "center", marginBottom: 20, marginTop: withinModal ? 24 : 20 }}>
                    <a className="button-link" onClick={handleLicenseSkip}>
                        {isPro ? i18n("licenseActivationSkipPro") : i18n("licenseActivationSkipFree")}
                    </a>
                </div>
            )}
        </>
    );
});

export { ConfigLicensingConsumer };
