import { Avatar, Card, Form, Space } from "antd";
import { observer } from "mobx-react";
import { useCallback, useEffect } from "react";

import { PluginUpdateEmbed, useStores as useRpmStores } from "@devowl-wp/real-product-manager-wp-client";

import { useStores } from "../../store/stores.js";
import { __ } from "../../utils/i18n.js";

import type { FC } from "react";

const ConfigLicensingConsumer: FC<{ withinModal?: boolean }> = observer(({ withinModal = false }) => {
    const {
        pluginUpdateStore: { pluginUpdates },
    } = useRpmStores();
    const { optionStore } = useStores();
    const {
        slug,
        others: { showLicenseFormImmediate },
        publicUrl,
    } = optionStore;
    const pluginUpdate = pluginUpdates.get(slug);

    const handleLicenseUpdated = useCallback(() => {
        // When form is shown directly after first usage, return to dashboard
        optionStore.setShowLicenseFormImmediate(false);
    }, [optionStore]);

    const handleLicenseSkip = useCallback(() => {
        optionStore.setShowLicenseFormImmediate(false);
        pluginUpdate.skip();
    }, [pluginUpdate, optionStore]);

    // Automatically skip if form already interacted (e.g. browser back cache issue)
    useEffect(() => {
        if (showLicenseFormImmediate && pluginUpdate?.hasInteractedWithFormOnce) {
            handleLicenseSkip();
        }
    }, [pluginUpdate, showLicenseFormImmediate, handleLicenseSkip]);

    return (
        <>
            {showLicenseFormImmediate && (
                <div style={{ maxWidth: 650, textAlign: "center", margin: "0 auto" }}>
                    {!withinModal && (
                        <Avatar
                            src={`${publicUrl}images/logos/real-physical-media.svg`}
                            shape="square"
                            size={130}
                            style={{ backgroundColor: "white", padding: 25, borderRadius: 999, marginBottom: 10 }}
                        />
                    )}
                </div>
            )}
            <Card
                title={withinModal || !pluginUpdate ? undefined : __("License activation")}
                bordered={!withinModal}
                bodyStyle={withinModal ? { paddingLeft: 0, paddingRight: 0 } : undefined}
            >
                <Space direction="vertical" size="large">
                    <PluginUpdateEmbed
                        formProps={{
                            onSave: handleLicenseUpdated,
                            footer: (
                                <Form.Item
                                    style={{
                                        margin: "25px 0 0",
                                        textAlign: showLicenseFormImmediate ? "center" : undefined,
                                    }}
                                >
                                    <input
                                        type="submit"
                                        className="button button-primary"
                                        value={
                                            showLicenseFormImmediate ? __("Activate license & continue") : __("Save")
                                        }
                                    />
                                </Form.Item>
                            ),
                        }}
                        listProps={{
                            onDeactivate: handleLicenseUpdated,
                        }}
                        slug={slug}
                    />
                </Space>
            </Card>
            {pluginUpdate && showLicenseFormImmediate && (
                <div
                    style={{ textAlign: "center", marginBottom: withinModal ? 0 : 20, marginTop: withinModal ? 0 : 20 }}
                >
                    <a className="button-link" onClick={handleLicenseSkip}>
                        {__("Continue without regular updates and without any support")}
                    </a>
                </div>
            )}
        </>
    );
});

export { ConfigLicensingConsumer };
