import { CopyOutlined, DeleteOutlined, RightOutlined, ScanOutlined } from "@ant-design/icons";
import { App, Layout, Spin } from "antd";
import { observer } from "mobx-react";
import { Fragment, useEffect, useState } from "react";

import { AnalyseResult } from "./analyseResult.js";
import { ProFooter, ProTag } from "./proFooter.js";
import { ThumbnailsList } from "./thumbnailsList.js";
import { useStores } from "../store/stores.js";
import { __, _n } from "../utils/i18n.js";

const { Header, Content, Footer } = Layout;

const SingleFileContent = observer(() => {
    const { message } = App.useApp();
    const { attachmentsStore, optionStore } = useStores();
    const [forceNewSchema, setForceNewSchema] = useState(false);
    const {
        others: { isPro, showProHints },
    } = optionStore;
    const { single, singleBusy } = attachmentsStore;

    useEffect(() => {
        setForceNewSchema(false);
    }, [single]);

    return (
        <Layout>
            <Header style={{ background: "#fff" }}>
                {__("Thumbnails")} <RightOutlined /> {__("Single file")}
            </Header>
            <Content style={{ margin: "24px 16px 0" }}>
                <div style={{ padding: 24, background: "#fff", minHeight: 360 }} className="rtg-single">
                    <Spin spinning={!!(singleBusy || single?.busy)}>
                        {single ? (
                            <div>
                                <h3>{single.filename}</h3>
                                <div className="clear" />
                                {!!single.medium && (
                                    <div className="rtg-size-img">
                                        <div>medium</div>
                                        <img src={single.medium} />
                                    </div>
                                )}
                                {!!single.thumbnail && (
                                    <div className="rtg-size-img">
                                        <div>thumbnail</div>
                                        <img src={single.thumbnail} />
                                    </div>
                                )}
                                <div className="clear" />
                                <AnalyseResult
                                    legend={<CopyOutlined />}
                                    style={{ display: "inline-block", marginRight: 10 }}
                                >
                                    <li>
                                        {_n("%d thumbnail", "%d thumbnails", single.cntGenerated, single.cntGenerated)}
                                    </li>
                                </AnalyseResult>
                                <AnalyseResult
                                    legend={<ScanOutlined />}
                                    style={{ display: "inline-block", marginRight: 10 }}
                                >
                                    <li>
                                        {_n(
                                            "%d thumbnail needs to be generated",
                                            "%d thumbnails need to be generated",
                                            single.newThumbnailsCount,
                                            single.newThumbnailsCount,
                                        )}
                                    </li>
                                </AnalyseResult>
                                <AnalyseResult
                                    legend={<DeleteOutlined />}
                                    style={{ display: "inline-block", marginRight: 10 }}
                                >
                                    <li>{__("%s unused", single.deletableSpaceFormat)}</li>
                                </AnalyseResult>
                                {single.usedSchema !== single.newSchema && (
                                    <div
                                        className="notice inline notice-info notice-alt"
                                        style={{ margin: "0 0 10px" }}
                                    >
                                        <p>
                                            {__(
                                                'The thumbnails were created before the last change of the thumbnail folder and file name structure. When using the "Regenerate" function, the old structure is kept, so that the image can still be found in existing posts (new structure would result in new image URL).',
                                            )}
                                            <br />
                                            <label>
                                                <input
                                                    type="checkbox"
                                                    checked={forceNewSchema}
                                                    onChange={() => setForceNewSchema(!forceNewSchema)}
                                                />
                                                {__("Force new thumbnail path and filename and delete old thumbnails")}
                                            </label>
                                            <br />
                                            {__("New schema:")} <code>{single.newSchema}</code>
                                        </p>
                                    </div>
                                )}
                                <div style={{ margin: "0 0 10px" }}>
                                    <button
                                        className="button button-primary"
                                        onClick={async () => {
                                            await single.regenerate(forceNewSchema, true);
                                            message.success(__("Successfully regenerated image!"));
                                        }}
                                    >
                                        {__("Regenerate")}
                                    </button>
                                    {!!single.error && (
                                        <div
                                            className="notice inline notice-error notice-alt"
                                            style={{ margin: "0 0 10px" }}
                                        >
                                            {single.error}
                                        </div>
                                    )}
                                    &nbsp;
                                    {(showProHints || isPro) && single.deletableSpace > 0 && (
                                        <Fragment>
                                            <button
                                                className="button"
                                                onClick={() => single.clear(true)}
                                                disabled={!isPro}
                                            >
                                                {__("Delete thumbnails of sizes that no longer exist")}
                                            </button>{" "}
                                            <ProTag feature="delete-unused-single" />
                                        </Fragment>
                                    )}
                                </div>
                                <ThumbnailsList />
                            </div>
                        ) : (
                            <div>
                                <h3>{__("No file selected.")}</h3>
                                <p>
                                    {__(
                                        "Please select an upload in the Analyze or bulk regenerate tab or open an upload in the media library to see details about the selected image.",
                                    )}
                                </p>
                            </div>
                        )}
                    </Spin>
                </div>
            </Content>
            {!isPro && showProHints && (
                <Footer>
                    <ProFooter />
                </Footer>
            )}
        </Layout>
    );
});

export { SingleFileContent };
