// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html
//
// CAssert.cpp : Implementation of CAssert

#include "stdafx.h"
#include "ScriptUnit.h"
#include "CAssert.h"
#include "TestEngine.h"



_variant_t DerefVariant(VARIANT& v)
{
	_variant_t var = v;
	if( var.vt == (VT_VARIANT|VT_BYREF) )
	{
		if( var.pvarVal )
			var = *var.pvarVal;
		else
			var.Clear();
	}
	return var;
}
/////////////////////////////////////////////////////////////////////////////
// CAssert

STDMETHODIMP CAssert::InterfaceSupportsErrorInfo(REFIID riid)
{
	static const IID* arr[] = 
	{
		&IID_IAssert
	};
	for (int i=0; i < sizeof(arr) / sizeof(arr[0]); i++)
	{
		if (InlineIsEqualGUID(*arr[i],riid))
			return S_OK;
	}
	return S_FALSE;
}

STDMETHODIMP CAssert::Equal(VARIANT v1, VARIANT v2, BSTR msg)
{
	ATLTRACE("CAssert::Equal\n");
	_variant_t var1 = DerefVariant(v1);
	_variant_t var2 = DerefVariant(v2);
	try {
		var1.ChangeType( var2.vt );
		if( var1 != var2 )
		{ 
			NotifyError("Assert.Equal failure", msg, var1, var2 );
			return E_FAIL;
		}
	}
	catch( ... )	// variant types not compatible
	{
		NotifyError("Assert.Equal failure", msg, var1, var2 );
		return E_FAIL;
	}

	return S_OK;
}

STDMETHODIMP CAssert::Equals(VARIANT v1, VARIANT v2, BSTR msg)
{
	return Equal(v1, v2, msg);
}

STDMETHODIMP CAssert::IsEqual(VARIANT v1, VARIANT v2, BSTR msg)
{
	return Equal(v1, v2, msg);
}


STDMETHODIMP CAssert::NotEqual(VARIANT v1, VARIANT v2, BSTR msg)
{
	ATLTRACE("CAssert::NotEqual\n");
	_variant_t var1 = DerefVariant(v1);
	_variant_t var2 = DerefVariant(v2);
	try {
		var1.ChangeType( var2.vt );
		if( var1 == var2 )
		{ 
			NotifyError("Assert.NotEqual failure", msg );
			return E_FAIL;
		}
	}
	catch( ... )	// variant types not compatible
	{
		NotifyError("Assert.NotEqual failure", msg );
		return E_FAIL;
	}
	return S_OK;
}

STDMETHODIMP CAssert::NotEquals(VARIANT v1, VARIANT v2, BSTR msg)
{
	return NotEqual(v1, v2, msg);
}

STDMETHODIMP CAssert::IsNotEqual(VARIANT v1, VARIANT v2, BSTR msg)
{
	return NotEqual(v1, v2, msg);
}

STDMETHODIMP CAssert::IsTrue(VARIANT_BOOL val, BSTR msg)
{
	ATLTRACE("CAssert::IsTrue\n");
	if( val == VARIANT_FALSE )
	{ 
		NotifyError("Assert.IsTrue failure", msg );
		return E_FAIL;
	}

	return S_OK;
}

STDMETHODIMP CAssert::IsFalse(VARIANT_BOOL val, BSTR msg)
{
	ATLTRACE("CAssert::IsFalse\n");
	if( val != VARIANT_FALSE )
	{ 
		NotifyError("Assert.IsFalse failure", msg );
		return E_FAIL;
	}

	return S_OK;
}

STDMETHODIMP CAssert::IsNothing(VARIANT val, BSTR msg)
{
	ATLTRACE("CAssert::IsNothing\n");
	_variant_t var = DerefVariant(val);
	try {
		if( var.vt == VT_NULL )
			return S_OK;

		var.ChangeType( VT_UNKNOWN );
		if( var.punkVal != NULL )
		{ 
			NotifyError("Assert.IsNothing failure", msg );
			return E_FAIL;
		}
	}
	catch( ... )	// variant is not an object at all
	{
			NotifyError("Assert.IsNothing - value is not an object", msg );
			return E_FAIL;
	}

	return S_OK;
}

STDMETHODIMP CAssert::IsNull(VARIANT val, BSTR msg)
{
	return IsNothing(val, msg);
}


STDMETHODIMP CAssert::IsNotNull(VARIANT val, BSTR msg)
{
	return IsSomething(val, msg);
}

STDMETHODIMP CAssert::IsSomething(VARIANT val, BSTR msg)
{
	ATLTRACE("CAssert::IsSomething\n");
	_variant_t var = DerefVariant(val);
	try {
		if( var.vt == VT_NULL )
		{
			NotifyError("Assert.IsSomething - value is null", msg );
			return E_FAIL;
		}

		var.ChangeType( VT_UNKNOWN );
		if( var.punkVal == NULL )
		{ 
			NotifyError("Assert.IsSomething failure", msg );
			return E_FAIL;
		}

	}
	catch( ... )	// variant is not an object at all
	{
			NotifyError("Assert.IsSomething - value is not an object", msg );
			return E_FAIL;
	}
	return S_OK;
}


STDMETHODIMP CAssert::ExpectError(/*[in,default]*/ BSTR errmsg)
{
	ATLTRACE("CAssert::ExpectError\n");
	_GotError = false;
	_ExpectingError = true;
	_ErrorMsg = errmsg;
	::CharLower((char*)_ErrorMsg);
	return S_OK;
}


bool CAssert::IsExpectingError(IScriptErrorPtr pse)
{ 
	ATLTRACE("CAssert::IsExpectingError?\n");
	_GotError = true;
	bool result = false;
	if( _ExpectingError && pse != NULL )
	{
		_bstr_t tmp = pse->GetDescription();
		::CharLower((char*)tmp);
		if( _ErrorMsg.length() == 0 || strstr(tmp, _ErrorMsg) > 0 )
			result = true;
	}
	// not expecting error and not having an error is also ok
	if( ! _ExpectingError && pse == NULL )
		result = true;

	return result; 
}

bool CAssert::IsNotExpectingError()
{ 
	ATLTRACE("CAssert::IsNotExpectingError?\n");
	bool result = ! _ExpectingError;
	return result; 
}


STDMETHODIMP CAssert::Trace(BSTR msg)
{
	ATLTRACE("CAssert::Trace\n");

	Notify("Trace", msg);

	return S_OK;
}

STDMETHODIMP CAssert::Remark(BSTR msg)
{
	ATLTRACE("CAssert::Remark\n");

	Notify("Remark", msg);

	return S_OK;
}


STDMETHODIMP CAssert::Error(/*[in,default]*/ BSTR errmsg)
{
	ATLTRACE("CAssert::Error\n");
	
	ClearExpectingError();
	NotifyError("Failure", errmsg);

	return E_FAIL;
}

STDMETHODIMP CAssert::Failure(/*[in,default]*/ BSTR errmsg)
{
	return Error(errmsg);
}


void CAssert::Notify(const _bstr_t& reason, const _bstr_t& msg )
{
	_bstr_t bsMsg = reason;
	bsMsg += ": ";
	bsMsg += msg;

	if( _pEngine )
		_pEngine->Trace( NULL, bsMsg );
}

void CAssert::NotifyError(const _bstr_t& reason, const _bstr_t& msg )
{
	Notify(reason, msg);
}

void CAssert::NotifyError(const _bstr_t& reason, const _bstr_t& msg, _variant_t& v1, _variant_t& v2 )
{
	_bstr_t diffs = msg;
	diffs += "\r\n";
	try {
		v1.ChangeType(VT_BSTR);
	}
	catch( ... )
	{
	}
	try {
		v2.ChangeType(VT_BSTR);
	}
	catch( ... )
	{
	}
	diffs += "'";
	if( v1.vt == VT_BSTR )
		diffs += v1.bstrVal;
	else
		diffs += "(???)";
	diffs += "' <> '";
	if( v2.vt == VT_BSTR )
		diffs += v2.bstrVal;
	else
		diffs += "(???)";
	diffs += "'";

	if( v1.vt == VT_BSTR || v2.vt == VT_BSTR )
		NotifyError( reason, diffs );
	else
		NotifyError( reason, msg );

}


