// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html

// TestDir.cpp: implementation of the CTestDir class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include <algorithm>
#include "TestFile.h"
#include "TestDir.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTestDir::CTestDir(const _bstr_t& path)
{
	_Path = path;
	_Status = kNew;
}

CTestDir::~CTestDir()
{
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		delete pFile;
	}

}

void CTestDir::SetStatus(EStatus s) 
{ 
	_Status = s; 
	if( ! _Item.IsNull() )
	{
		_Item.SetState(INDEXTOSTATEIMAGEMASK(_Status), TVIS_STATEIMAGEMASK);
	}
}

void CTestDir::UpdateStatus() 
{ 
	EStatus s = kUnknown;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile->GetStatus() == kFailure )
			s = kFailure;
		if( pFile->GetStatus() == kSuccess && s != kFailure )
			s = kSuccess;
	}
	SetStatus( s );
}



_bstr_t CTestDir::GetPath()
{
	return _Path;
}

_bstr_t CTestDir::GetName()
{
	_bstr_t name = "???";

	const char* pPath = _Path;
	char* pSlash = strrchr(pPath, '\\');
	if( pSlash )
		name = pSlash;
	else
		name = _Path;
	return name;
}

CTestFile* CTestDir::AddFile(CTestFile* pFile)
{
	CTestFile* pResult = NULL;
	// need to compare names, not addersses
	FileMap::iterator i = _Files.find( pFile->GetName() );

	// add the file if it isn't already in the list
	if( i == _Files.end() )
	{
		_Files[ pFile->GetName() ] = pFile;
		pResult = pFile;
	}
	return pResult;
}


void CTestDir::RemoveFile(CTestFile* pFile)
{
	if( pFile )
	{
		if( ! pFile->GetItem().IsNull() )
			pFile->GetItem().Delete();
		_Files.erase( pFile->GetName() );
	}
}


FileMap::iterator CTestDir::GetFilesBegin()
{
	return _Files.begin();
}

FileMap::iterator CTestDir::GetFilesEnd()
{
	return _Files.end();
}


long CTestDir::NumFiles()
{
	long n = 0;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile )
			++n;
	}
	return n;
}

long CTestDir::NumTests()
{
	long n = 0;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile )
			n += pFile->NumTests();
	}
	return n;
}

long CTestDir::NumFail()
{
	long n = 0;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile )
			n += pFile->NumFail();
	}
	return n;
}

long CTestDir::NumNotRun()
{
	long n = 0;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile )
			n += pFile->NumNotRun();
	}
	return n;
}

long CTestDir::NumTicks()
{
	long n = 0;
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile )
			n += pFile->NumTicks();
	}
	return n;
}

_bstr_t CTestDir::GetTrace()
{
	bool hasFailed = false;
	_bstr_t trace = "Failures:\r\n";
	for( FileMap::iterator i = _Files.begin(); i != _Files.end(); ++i )
	{
		CTestFile* pFile = i->second;
		if( pFile->GetStatus() == kFailure )
		{
			trace += pFile->GetTrace();
			trace += "\r\n";
			hasFailed = true;
		}
	}
	if( ! hasFailed )
		trace = "No failures!\r\n";
	return trace;
}

