// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html

#include "stdafx.h"
#include "TestEngine.h"
#include "TestFile.h"
#include "TestDir.h"

#include "atl_enum_iterator.h"





#define RELEASEANDCLEAR(x)		\
	if( x ) x->Release();		\
	x = NULL;

CTestEngine::CTestEngine()
{
	_pAssert	 = NULL;
	_pEngine	 = NULL;
	_curFile	 = NULL;
	_curFixture  = NULL;
	_pFso.CreateInstance( "Scripting.FileSystemObject" );
	_pScriptCtrl = NULL;
}


CTestEngine::~CTestEngine()
{
::OutputDebugString("~CTestEngine dtor\n");
	_pAssert = NULL;
	RELEASEANDCLEAR( _pAssert ); 
	RELEASEANDCLEAR( _pEngine ); 
	_pFso        = NULL;
	_pAssert     = NULL;
	_curFile	 = NULL;
	_curFixture  = NULL;
}



bool CTestEngine::IsScript(const _bstr_t& ext)
{
	bool isScript = false;


	HKEY hExtKey = NULL;
	HKEY hTypeKey = NULL;
	HRESULT res = RegOpenKey( HKEY_CLASSES_ROOT, (char*)ext, &hExtKey );
	if( SUCCEEDED( res ) )
	{
		char bufTypeName[100];
		long cbBuf = 100;
		res = RegQueryValue(hExtKey, NULL, bufTypeName, &cbBuf);
		if( res == ERROR_SUCCESS && cbBuf > 0 )
		{
			res = RegOpenKey( HKEY_CLASSES_ROOT, bufTypeName, &hTypeKey );
			if( SUCCEEDED( res ) )
			{
				char bufEngineName[100];
				cbBuf = 100;
				res = RegQueryValue(hTypeKey, "ScriptEngine", bufEngineName, &cbBuf);
				if( res == ERROR_SUCCESS && cbBuf > 0 )
				{
					isScript = true;
					_bstr_t engineName = bufEngineName;
					_ScriptEngines[ext] = engineName;
				}
				RegCloseKey(hTypeKey);
			}
		
		}
		RegCloseKey(hExtKey);
	}

	return isScript;
}

void CTestEngine::InitFile(CTestFile* pFile)
{
	assert(pFile);

	StartTrace( pFile );	// set current file for trace logging
	Trace( pFile, pFile->GetPath() );
	pFile->SetStatus( kRunning );
	InitEngine( pFile->GetExt() );

	if( _pScriptCtrl->Language.length() == 0 )
	{
		Trace( pFile, L"Unable to load scripting engine for file");
		Trace( pFile, pFile->GetType() );
		pFile->SetStatus( kFailure );
		return;
	}

	try {
		_bstr_t tmp = pFile->GetCode();
		_pScriptCtrl->AddCode( tmp );
		if( ! _pAssert->IsNotExpectingError() )
		{
			Trace( pFile, L"Error while loading script file - expecting error that did not come" );
			pFile->SetStatus( kFailure );
		}

	}
	catch( _com_error& e )
	{
		IScriptErrorPtr pe = _pScriptCtrl->GetError();
		if( ! _pAssert->IsExpectingError(pe) )
		{
			Trace( pFile, L"Error while loading script file" );
			Trace( pFile, e );
			Trace( pFile, pe );
			pFile->SetStatus( kFailure );
		}
		else
		{
			pFile->SetStatus( kSuccess );
		}
	}
}

void CTestEngine::InitEngine(const _bstr_t& ext)
{
::OutputDebugString("CTestEngine::INIT\n");
	try 
	{
		_pScriptCtrl = NULL;
		RELEASEANDCLEAR( _pAssert ); 
		RELEASEANDCLEAR( _pEngine ); 

		_pScriptCtrl.CreateInstance( "MSScriptControl.ScriptControl" );
		_pScriptCtrl->PutAllowUI( VARIANT_TRUE );
		_pScriptCtrl->PutUseSafeSubset( VARIANT_FALSE );
		_pScriptCtrl->PutTimeout( 30000 );

		// look up script engine name thru the registry - _ScriptEngines initialized in IsScript function...
		_bstr_t lang = _ScriptEngines[ ext ];
		_pScriptCtrl->PutLanguage( lang );

		// no valid scripting engine found
		if( _pScriptCtrl->GetLanguage().length() == 0 )
			return;

		CComObject<CAssert>::CreateInstance( & _pAssert );
		_pAssert->AddRef();
		CComObject<CEngine>::CreateInstance( & _pEngine );
		_pEngine->AddRef();
		_pEngine->Init( _pScriptCtrl );

		ScriptControlStates initState = _pScriptCtrl->GetState();
 
		_pAssert->SetEngine( this );
		_pScriptCtrl->AddObject("Assert", _pAssert, VARIANT_TRUE );
		_pScriptCtrl->AddObject("ScriptUnit", _pEngine, VARIANT_FALSE );
	}
	catch( ... )
	{
		_pScriptCtrl->put_Language( L"" );
		throw;
	}
}





int CTestEngine::RunAllTests()
{
	long numTests = 0;
	DirMap::iterator i;
	for( i = _Dirs.begin(); i != _Dirs.end(); ++i )
		numTests += i->second->NumTests();

	GBroker::Notify( this, N_TESTSTART, numTests );

	for( i = _Dirs.begin(); i != _Dirs.end(); ++i )
	{
		CTestDir* pDir = i->second;
		RunTest(pDir);
	}

	GBroker::Notify( this, N_TESTFINISH, 0 );
	return 0;
}

void CTestEngine::RunThisTest(CTestDir* pDir)
{
	if( pDir )
	{
		GBroker::Notify( this, N_TESTSTART, pDir->NumTests()  );
		RunTest( pDir );
		GBroker::Notify( this, N_TESTFINISH, 0 );
	}
}

void CTestEngine::RunThisTest(CTestFile* pFile)
{
	if( pFile )
	{
		// add one for the file itself
		GBroker::Notify( this, N_TESTSTART, pFile->NumTests() );
		RunTest( pFile );
		GBroker::Notify( this, N_TESTFINISH, 0);
	}
}

void CTestEngine::RunThisTest(CTestFixture* pFixture)
{
	if( pFixture )
	{
		// add one for the file notification
		GBroker::Notify( this, N_TESTSTART, pFixture->NumTests() );
		RunTest( pFixture );
		GBroker::Notify( this, N_TESTFINISH, 0 );
	}
}





void CTestEngine::RunTest(CTestDir* pDir)
{
	_Break = false;
	if( pDir ) 
	{
		pDir->SetStatus( kRunning );
		for( FileMap::iterator j = pDir->GetFilesBegin(); j != pDir->GetFilesEnd(); ++j )
		{
			CTestFile* pFile = j->second;
			RunTest(pFile);
			if( _Break )
				return;
			if( pFile->GetStatus() == kFailure )
				pDir->SetStatus( kFailure );
		}
		if( pDir->GetStatus() == kRunning )
			pDir->SetStatus( kSuccess );
	}
}

void CTestEngine::RunTest(CTestFile* pFile)
{
	_Break = false;
	if( pFile )
	{
		// scan file for new fixtures first
		UpdateFixtures(pFile);	

		StartTrace( pFile );
		long timeStart = GetTickCount();
		InitFile( pFile );
		long timeEnd = GetTickCount();
		GBroker::Notify( this, N_TESTSTEP, 1 );	// mainline counts as one
		pFile->SetNumTicks( Duration(timeStart, timeEnd) );
		
		if( pFile->GetStatus() == kFailure )
			return;

		try {
			// now call the callable TestXxx subs
			for( FixtureMap::iterator j = pFile->GetFixturesBegin(); j != pFile->GetFixturesEnd(); ++j )
			{
				CTestFixture* pFixture = j->second;
				RunFixture(pFixture);
				GBroker::Notify( this, N_TESTSTEP, 1 );	// test counts as one step
				if( _Break )
					return;
				if( pFixture->GetStatus() == kFailure )
					pFile->SetStatus( kFailure );
			}
			if( pFile->GetStatus() != kFailure )
				pFile->SetStatus( kSuccess );
			else
				pFile->SetStatus( kFailure );
	
		}
		catch( _com_error& e )
		{
			Trace( pFile, L"Script error while running script" );
			Trace( pFile, e );
		}
	}
	return;
}

void CTestEngine::RunTest(CTestFixture* pFixture)
{
	_Break = false;
	if( pFixture )
	{
		StartTrace( pFixture );	// set current file for trace logging
		pFixture->SetStatus( kRunning );
		InitFile( pFixture->GetFile() );
		if( pFixture->GetStatus() == kFailure )
			return;
		RunFixture(pFixture);
		GBroker::Notify( this, N_TESTSTEP, 1 );	// test counts as one step
	}
	return;
}


void CTestEngine::RunFixture(CTestFixture* pFixture)
{
	_curFixture = pFixture;
	pFixture->SetStatus( kRunning );
	
	DWORD timeStart, timeEnd;
	timeStart = GetTickCount();


	SAFEARRAYBOUND abound;
	abound.cElements = 0;
	abound.lLbound   = 0;
	SAFEARRAY* pArray = SafeArrayCreate(VT_VARIANT,1,&abound);
	if(! pArray)
	{
		Trace( pFixture, L"Unable to run test fixture - cannot set up call to test method" );
		pFixture->SetStatus( kFailure );
		return;
	}

	if( pFixture->GetFile()->HasSetup() )
	{
		try 
		{ 
			_pAssert->ClearExpectingError();
			_pScriptCtrl->Error->Clear();
			_pScriptCtrl->Run( "Setup", &pArray ); 
		} 
		catch( _com_error& e )
		{
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Error while running Setup function" );
				Trace( pFixture, e );
				Trace( pFixture, pe );
			}
		}
		catch (...) 
		{ 
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Unknown error while running Setup function" );
				Trace( pFixture, pe );
			}
		}
	}
	if( pFixture->GetStatus() == kRunning )
	{
		try 
		{ 
			_pAssert->ClearExpectingError();
			_pScriptCtrl->Error->Clear();
			_pScriptCtrl->Run( pFixture->Name(), &pArray  );
			if( _pAssert->IsNotExpectingError() )
			{
				pFixture->SetStatus( kSuccess );
			}
			else
			{
				Trace( pFixture, L"Error while running script - expecting error that did not come" );
				pFixture->SetStatus( kFailure );
			}
		} 
		catch( _com_error& e )
		{
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Error while running script" );
				Trace( pFixture, e );
				Trace( pFixture, _pScriptCtrl->GetError() );
			}
			else
				pFixture->SetStatus( kSuccess );
		}
		catch (...) 
		{ 
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Unknown error while running script" );
				Trace( pFixture, pe );
			}
			else
				pFixture->SetStatus( kSuccess );
		}
	}

	if( pFixture->GetFile()->HasTeardown() )
	{
		try { 
			if( pFixture->GetStatus() == kFailure )
			{
				_pAssert->ClearExpectingError();
				_pScriptCtrl->Error->Clear();
			}
			_pScriptCtrl->Run( "Teardown", &pArray );
		} 
		catch( _com_error& e )
		{
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Error while running Teardown function" );
				Trace( pFixture, e );
				Trace( pFixture, pe );
			}
		}
		catch (...) 
		{ 
			IScriptErrorPtr pe = _pScriptCtrl->GetError();
			if( ! _pAssert->IsExpectingError(pe) )
			{
				pFixture->SetStatus( kFailure );
				Trace( pFixture, L"Unknown error while running Teardown function" );
				Trace( pFixture, pe );
			}
		}
	}
	SafeArrayDestroy(pArray);
	pArray = NULL;

	timeEnd = GetTickCount();

	pFixture->SetNumTicks( Duration(timeStart, timeEnd) );

}

void CTestEngine::StopTest()
{
	_Break = true;
	if( _pScriptCtrl )
	{
		// _pScriptCtrl->Reset();
	}
}


void CTestEngine::RemoveFixture(CTestFixture* pFixture)
{
	if( pFixture )
	{
		CTestFile* pFile = pFixture->GetFile();
		if( pFile )
		{
			pFile->RemoveFixture( pFixture );
			delete pFixture;
		}
	}
}

void CTestEngine::RemoveFile(CTestFile* pFile)
{
	if( pFile )
	{
		CTestDir* pDir = pFile->GetDir();
		if( pDir )
		{
			pDir->RemoveFile( pFile );
			delete pFile;
		}
	}
}

void CTestEngine::RemoveDir(CTestDir* pDir)
{
	if( pDir )
	{
		FileMap::iterator j = pDir->GetFilesBegin();
		while( j != pDir->GetFilesEnd() )
		{
			CTestFile* pFile = j->second;
			pDir->RemoveFile( pFile );
			j = pDir->GetFilesBegin();
		}
		if( ! pDir->GetItem().IsNull() )
			pDir->GetItem().Delete();
		_Dirs.erase( pDir->GetPath() );
		delete pDir;
	}
}


void CTestEngine::UpdateFixtures(CTestFile* pFile)
{
	InitFile( pFile );
	pFile->SetSetup(false);
	pFile->SetTeardown(false);
	if( pFile->GetStatus() == kFailure )
		return;

	for( FixtureMap::iterator k = pFile->GetFixturesBegin(); k != pFile->GetFixturesEnd(); ++k )
	{
		CTestFixture* pFixture = k->second;
		pFixture->SetStatus( kDelete );
	}

	// enumerate through the collection using the lovely IEnumVariant API. (blech!)
	IScriptProcedureCollectionPtr pProcs = _pScriptCtrl->Procedures;
	IUnknown* pUnk = NULL;
	pProcs->get__NewEnum(&pUnk);
	IEnumVARIANT* pEnumVar = NULL;
	if( pUnk != NULL )
	{
		pUnk->QueryInterface(IID_IEnumVARIANT, (void**)&pEnumVar);
		if( pEnumVar )
		{
			ULONG nFetched = 1;
			while( nFetched )
			{
				VARIANT elem;
				VariantInit(&elem);
				nFetched = 0;
				pEnumVar->Next(1, &elem, &nFetched);
				_variant_t vElem(elem);

				if( nFetched && vElem.vt == VT_DISPATCH )
				{
					IScriptProcedurePtr pProc;
					IDispatchPtr pDisp = (IDispatch*) vElem;
					pProc = pDisp;
					VariantClear(&elem);

					if( pProc != NULL )
					{
						_bstr_t theName = pProc->GetName();
						if( strnicmp(theName, "Setup", 5) == 0 && pProc->GetNumArgs() == 0 )
							pFile->SetSetup(true);
						if( strnicmp(theName, "Teardown", 8) == 0 && pProc->GetNumArgs() == 0 )
							pFile->SetTeardown(true);
						if( strnicmp(theName, "Test", 4) == 0 && pProc->GetNumArgs() == 0 )
						{
							CTestFixture* pNewFixture = pFile->GetFixture( theName );
							if( pNewFixture == NULL )
							{
								pNewFixture = new CTestFixture( pFile, theName );
								pFile->AddFixture( pNewFixture );
								pNewFixture->SetStatus( kNew );
								GBroker::Notify( pNewFixture, N_ADDFIXTURE, 0 );					
							}
							else
							{
								pNewFixture->SetStatus( kUnknown );
							}
						}
					}
				}
			}
			RELEASEANDCLEAR(pEnumVar);
		}
		RELEASEANDCLEAR(pUnk);	
	}

	// remove items that don't exist any more
	for( FixtureMap::iterator j = pFile->GetFixturesBegin(); j != pFile->GetFixturesEnd(); ++j )
	{
		CTestFixture* pFixture = j->second;
		if( pFixture->GetStatus() == kDelete )
		{
			GBroker::Notify( pFixture, N_REMOVEFIXTURE, 0 );
			pFile->RemoveFixture( pFixture );
			j = pFile->GetFixturesBegin(); 
		}
	}
}

CTestFile* CTestEngine::AddFile(const _bstr_t& filename)
{
	CTestFile* pFile = NULL;
	try 
	{
		IFilePtr pIFile = _pFso->GetFile( filename );
		FileAttribute attr = pIFile->GetAttributes();
		if( (attr & (Hidden | System | Volume)) == 0 )
		{
			// this is a file - load it in
			IFolderPtr pFolder = pIFile->GetParentFolder();
			CTestDir* pDir = GetDir( pFolder->GetPath() );

			CTestFile* pTestFile = new CTestFile( pDir, pIFile->GetPath(), pIFile->GetType() );
			if( IsScript( pTestFile->GetExt() ) )
			{
				pFile = pDir->AddFile( pTestFile );
				if( pFile == NULL )
					delete pTestFile;
				else
				{
					if( ! pTestFile->GetItem().IsNull() )
					{
//						UpdateFixtures(pTestFile);
					}
					pTestFile->SetStatus( kNew );
				}
			}
			else
				delete pTestFile;
		}
	}
	catch (...)
	{
		// bad filename - ignore it
	}

	return pFile;
}

CTestDir* CTestEngine::AddDir(const _bstr_t& filename)
{
	CTestDir*  pDir = NULL;
	CTestFile* pFile = NULL;
	try 
	{
		// load all files in dir into engine
		IFolderPtr pFolder = _pFso->GetFolder( filename );
		IFileCollectionPtr pFiles = pFolder->GetFiles();
		long n = pFiles->GetCount();

		IUnknown* pevar;
		pFiles->get__NewEnum(&pevar);
		for( variant_iterator i = variant_iterator(pevar, 64); i != variant_iterator(); ++i )
		{
			VARIANT&    v = *i;
			if( v.vt == VT_DISPATCH )
			{
				IFile* pIFile;
				IFilePtr spIFile;
				v.punkVal->QueryInterface( __uuidof(IFile), (void**) &pIFile );
				spIFile.Attach(pIFile);
				pFile = AddFile( spIFile->GetPath() );
			}
		}
		pevar->Release();
		pevar = NULL;

	}
	catch (...)
	{
		// bad filename - ignore it
	}
	if( pFile )
		pDir = pFile->GetDir();
	return pDir;
}


bool CTestEngine::IsDir(const _bstr_t& filename)
{
	try 
	{
		return _pFso->FolderExists( filename ) != VARIANT_FALSE;
	}
	catch (...)
	{
		// bad filename - ignore it
	}
	return false;
}


CTestDir* CTestEngine::GetDir(const _bstr_t& path)
{
	CTestDir* pDir = NULL;
	pDir = _Dirs[path];
	if( pDir == NULL )
	{
		pDir = new CTestDir(path);
		_Dirs[path] = pDir;
	}
	return pDir;
}


DirMap::iterator CTestEngine::GetDirsBegin()
{
	return _Dirs.begin();
}

DirMap::iterator CTestEngine::GetDirsEnd()
{
	return _Dirs.end();
}


void CTestEngine::StartTrace(CTestFile* pFile)
{
	// output a separator line to the trace log
	_curFile = NULL;
	_curFixture = NULL;
	Trace(pFile, L"-------------------------------------------------------------------------");

	_curFile = pFile;
	pFile->ResetTrace();
	for( FixtureMap::iterator j = pFile->GetFixturesBegin(); j != pFile->GetFixturesEnd(); ++j )
	{
		CTestFixture* pFixture = j->second;
		pFixture->ResetTrace();
	}

}


void CTestEngine::StartTrace(CTestFixture* pFixture)
{
	_curFixture = pFixture;
	_curFile = pFixture->GetFile();
	_curFile->ResetTrace();
	_curFixture->ResetTrace();
}


void CTestEngine::SaveTrace(const _bstr_t& filename)
{
	if( filename.length() > 0 )
	{
		long numTests, numFail, numNotRun, numTicks;
		numTests = numFail = numNotRun = numTicks = 0;
		DirMap::iterator i;
		FileMap::iterator j;
		FixtureMap::iterator k;
		for( i = _Dirs.begin(); i != _Dirs.end(); ++i )
		{
			numTests += i->second->NumTests();
			numFail  += i->second->NumFail();
			numNotRun+= i->second->NumNotRun();
			numTicks += i->second->NumTicks();
		}
		// non-localized date format hack
		SYSTEMTIME t;
		GetSystemTime(&t);
		char theDate[80], theTotalTime[80];
		sprintf(theDate, "%d.%d.%d", t.wDay, t.wMonth, t.wYear);
		sprintf(theTotalTime, "%d:%02d", numTicks / 60000, (numTicks % 60000) / 1000 );
		
		FILE* f = fopen( filename, "w+t" );
		fprintf( f, "<?xml version=\"1.0\" standalone=\"no\" ?>\n", f );
		fprintf( f, "<!-- This file represents the results of running a test suite in ScriptUnit -->\n" );
		fprintf( f, "<test-results name=\"%s\" total=\"%d\" failures=\"%d\" not-run=\"%d\" date=\"%s\" time=\"%s\">",
					(char*)filename, numTests, numFail, numNotRun, theDate, theTotalTime );
		for( i = _Dirs.begin(); i != _Dirs.end(); ++i )
		{
			CTestDir* pDir = i->second;
			fprintf( f, "<test-suite name=\"%s\" success=\"%s\" time=\"%f\">\n",
				(char*) pDir->GetPath(), ISSUCCESS(pDir->GetStatus()), TicksToSeconds(pDir->NumTicks()) );
			fprintf( f, "<results>\n");

			for( j = pDir->GetFilesBegin(); j != pDir->GetFilesEnd(); ++j )
			{
				CTestFile* pFile = j->second;
				fprintf( f, "<test-suite name=\"%s\" success=\"%s\" time=\"%f\">\n",
					(char*) pFile->GetName(), ISSUCCESS(pFile->GetStatus()), TicksToSeconds(pFile->NumTicks()) );
				fprintf( f, "<results>\n");

				if( pFile->NumFixtures() == 0 )
				{
					fprintf( f, "<test-case name=\"%s\" executed=\"%s\" success=\"%s\" time=\"%f\">",
						(char*) PrettyTestCaseName(pFile, NULL), ISRUN(pFile->GetStatus()), ISSUCCESS(pFile->GetStatus()), TicksToSeconds(pFile->NumTicks()) );
					if( pFile->GetStatus() == kFailure )
					{
						fprintf( f, "\n<failure>\n" );
						fprintf( f, "<message><![CDATA[" );
						fprintf( f, pFile->GetTrace() );
						fprintf( f, "]]></message>\n" );
						fprintf( f, "<stack-trace><![CDATA[" );
						fprintf( f, pFile->GetInfo() );
						fprintf( f, "]]></stack-trace>\n" );
						fprintf( f, "</failure>\n" );
					}
					fprintf( f, "</test-case>\n");
				}

				for( k = pFile->GetFixturesBegin(); k != pFile->GetFixturesEnd(); ++k )
				{
					CTestFixture* pFix = k->second;
					fprintf( f, "<test-case name=\"%s\" executed=\"%s\" success=\"%s\" time=\"%f\" >",
						(char*) PrettyTestCaseName(pFile, pFix), ISRUN(pFix->GetStatus()), ISSUCCESS(pFix->GetStatus()), TicksToSeconds(pFix->NumTicks()) );
					if( pFix->GetStatus() == kFailure )
					{
						fprintf( f, "\n<failure>\n" );
						fprintf( f, "<message><![CDATA[" );
						fprintf( f, pFix->GetTrace() );
						fprintf( f, "]]></message>\n" );
						fprintf( f, "<stack-trace><![CDATA[" );
						fprintf( f, pFix->GetInfo() );
						fprintf( f, "]]></stack-trace>\n" );
						fprintf( f, "</failure>\n" );
					}
					fprintf( f, "</test-case>\n");
				}
				fprintf( f, "</results>\n");
				fprintf( f, "</test-suite>\n");
			}

			fprintf( f, "</results>\n");
			fprintf( f, "</test-suite>\n");
		}
		fprintf( f, "</test-results>\n");
		fclose( f );
	}

}

void CTestEngine::Trace(CTestBase* pTest, const _bstr_t& msg)
{
	if( msg.length() > 0 && msg.length() < 512)
		ATLTRACE( (char*) msg );
	ATLTRACE( "\n" );

	// output to file?
	// notify file
	if( pTest == NULL ) 
		pTest = _curFixture;
	if( pTest == NULL ) 
		pTest = _curFile;

	if( pTest )
		if( msg.length() > 0 )
		{
			pTest->AddTrace( msg );
			pTest->AddTrace( "\r\n" );
		}
}

void CTestEngine::Trace(CTestBase* pTest, const _com_error& e)
{
	_bstr_t errmsg = e.Description();
	Trace( pTest, errmsg );
	errmsg = e.ErrorMessage();
	Trace( pTest, errmsg );
}

void CTestEngine::Trace(CTestBase* pTest, IScriptErrorPtr pe)
{
	if( pe != NULL && pTest != NULL )
	{
		char buf[200];
		if( ((char*)pe->Source) == NULL )
		{
			sprintf(buf, "Error: Line %d  Column %d", pe->Line, pe->Column);
			pTest->SetFaultLine( pe->Line );
		}
		else
		{
			sprintf(buf, "Error: Source: %s  Line %d  Column %d", (char*) pe->Source, pe->Line, pe->Column);
			pTest->SetFaultLine( 0 );	// the line is not in main file
		}

		Trace( pTest, _bstr_t(buf) );
		Trace( pTest, pe->Description );
		Trace( pTest, pe->Text );
	}
}

void CTestEngine::LoadSettings()
{
	HKEY hKey = NULL;
	HRESULT res = RegOpenKey( HKEY_CURRENT_USER, "Software\\ScriptUnit", &hKey );
	if( SUCCEEDED( res ) )
	{
		DWORD i = 0;
		while( ERROR_SUCCESS == res )
		{
			char name[30];
			char buf[300];
			DWORD cbName = 30;
			DWORD cbBuf  = 300;
			DWORD type = 0;
			res = ::RegEnumValue( hKey, i, name, &cbName, NULL, &type, (LPBYTE) &buf, &cbBuf );
			if( ERROR_SUCCESS == res && type == REG_SZ )
			{
				_bstr_t filepath = buf;
				CTestFile* pFile = this->AddFile(filepath);
				if( pFile )
					UpdateFixtures(pFile);

			}
			++i;
		}
		RegCloseKey(hKey);
	}

}

void CTestEngine::SaveSettings()
{
	// delete old settings first
	RegDeleteKey( HKEY_CURRENT_USER, "Software\\ScriptUnit" );

	// now save the new settings under the same key
	HKEY hKey = NULL;
	HRESULT res = RegCreateKey( HKEY_CURRENT_USER, "Software\\ScriptUnit", &hKey );
	if( SUCCEEDED( res ) )
	{
		int i = 0;
		for( DirMap::iterator d = GetDirsBegin(); d != GetDirsEnd(); ++d )
		{
			CTestDir* pDir = d->second;
			for( FileMap::iterator f = pDir->GetFilesBegin(); f != pDir->GetFilesEnd(); ++f, ++i )
			{
				CTestFile* pFile = f->second;
				char name[30];
				sprintf(name, "File%4d", i);
				::RegSetValueEx( hKey, name, 0, REG_SZ, (BYTE*) (char*) (pFile->GetPath()), pFile->GetPath().length()+1 );
			}
		}
		RegCloseKey(hKey);
	}
}

long CTestEngine::Duration(const DWORD start, const DWORD end)
{
	long numTicks = 0;
	if( end >= start )
		numTicks = end - start + 1;
	else
		numTicks = start + (DWORD(-1) - end);
	return numTicks;
}


_bstr_t CTestEngine::PrettyTestCaseName(CTestFile* pFile, CTestFixture* pFix)
{
	_bstr_t result;
	if(pFile)
	{
		_bstr_t name = pFile->GetPath().copy();;
		char* pName = name;
		if( pName[1] == ':' )		// skip leading drive letter and colon
			pName+= 2;
		while( pName[0] == '\\' )		// skip leading backslashes (UNC and root marker)
			pName++;
		char* pCh = pName;
		while( pCh[0] != 0 )			// convert backslash to dot
		{
			if( *pCh == '\\' )
				*pCh = '.';
			pCh++;
		}
		// convert foobar.vbs --> foobar_vbs
		char* pLastDot = strrchr(pName, '.');
		if( pName && pLastDot )
			*pLastDot = '_';

		result = pName;

		if( pFix )
		{
			result += ".";
			result += pFix->Name();
		}
	}
	return result;
}