// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html

/*
 * COM Tester main engine
 * 
 * Loads scripts, interrogates them for tests
 * figures out what to display
 *
 * Allows tests to be run, and returns pass/fail info
 *
*/

#ifndef TEST_ENGINE_H
#define TEST_ENGINE_H

#include <map>
#include "Observer.h"
#include "CAssert.h"
#include "CEngine.h"
#include "TestFixture.h"
#include "TestFile.h"
#include "TestDir.h"

#include "ComWrappers.h"	// IScriptControlPtr

typedef std::map<_bstr_t, _bstr_t>		NameMap;
typedef std::map<_bstr_t, CTestDir*>	DirMap;


class CTestEngine : public ISubject
{
public:
	CTestEngine();
	virtual ~CTestEngine();

	void		InitEngine(const _bstr_t& ext);
	void		InitFile(CTestFile* pFile);

	CTestFile*	AddFile(const _bstr_t& filename);
	CTestDir*	AddDir(const _bstr_t& filename);
	CTestDir*	GetDir(const _bstr_t& path);
	void		RemoveFixture(CTestFixture* pFixture);
	void		RemoveFile(CTestFile* pFile);
	void		RemoveDir(CTestDir* pDir);
	bool		IsDir(const _bstr_t& filename);	

	int  RunAllTests();
	void RunThisTest(CTestDir* pDir);
	void RunThisTest(CTestFile* pFile);
	void RunThisTest(CTestFixture* pFixture);
	void StopTest();

	void SaveTrace(const _bstr_t& filename);
	void StartTrace(CTestFile* pFile);
	void StartTrace(CTestFixture* pFixture);
	void Trace(CTestBase* pTest, const _bstr_t& msg);
	void Trace(CTestBase* pTest, const _com_error& e);
	void Trace(CTestBase* pTest, IScriptErrorPtr pe);
	void Trace(CTestBase* pTest, const BSTR msg) { Trace(pTest, _bstr_t(msg) ); }

	void LoadSettings();
	void SaveSettings();

	DirMap::iterator GetDirsBegin();
	DirMap::iterator GetDirsEnd();

	void UpdateFixtures(CTestFile* pFile);

protected:
	// recursive calls
	void RunTest(CTestDir* pDir);
	void RunTest(CTestFile* pFile);
	void RunTest(CTestFixture* pFixture);

	// run StartupFixture/TeardownFixture
	void RunFixture(CTestFixture* pFixture);

private:
	IFileSystemPtr	_pFso;
	IScriptControlPtr _pScriptCtrl;	// for running scripts
	CAssertPtr		_pAssert;		// "Assert" object for scripts to use
	CEnginePtr		_pEngine;		// "ScriptUnit" object for scripts to use
	DirMap			_Dirs;
	CTestFile*		_curFile;
	CTestFixture*	_curFixture;
	bool			_Break;			// has break button been pressed?
	NameMap			_ScriptEngines;	// map .ext to scripting engine name

	bool IsScript(const _bstr_t& ext);

	// for logging to XML file
	long Duration(const DWORD start, const DWORD end);
	double TicksToSeconds(long ticks) { return ticks / 1000.0; }
	_bstr_t PrettyTestCaseName(CTestFile* pFile, CTestFixture* pFix);
};

#endif