// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html

// TestFile.cpp: implementation of the CTestFile class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "TestFixture.h"
#include "TestFile.h"
#include "TestDir.h"
#include "Observer.h"

#include <string.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTestFile::CTestFile(CTestDir* pDir, const _bstr_t& path, const _bstr_t& type)
{
	_pDir = pDir;
	_Path = path;
	_Type = type;
	_Status = kNew;
	_Fixtures.clear();

	_Setup = false;
	_Teardown = false;
	_NumTicks = 0;
}

CTestFile::~CTestFile()
{
	_pDir = NULL;
	for( FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFixture = i->second;
		delete pFixture;
	}
	_Fixtures.clear();
	_NumTicks = 0;
}


void CTestFile::SetStatus(EStatus s) 
{ 
	if( s != _Status )
	{
		_Status = s; 
		if( ! _Item.IsNull() )
			_Item.SetState(INDEXTOSTATEIMAGEMASK(_Status), TVIS_STATEIMAGEMASK);
		GBroker::Notify( this, N_TESTSTATUS, s );
		_pDir->UpdateStatus();
	}
}


void CTestFile::UpdateStatus() 
{ 
	EStatus s = kUnknown;
	for( FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFixture = i->second;
		if( pFixture->GetStatus() == kFailure )
			s = kFailure;
		if( pFixture->GetStatus() == kSuccess && s != kFailure )
			s = kSuccess;
	}
	SetStatus( s );
}



CTestDir* CTestFile::GetDir()
{
	return _pDir;
}

_bstr_t CTestFile::GetPath()
{
	return _Path;
}

_bstr_t CTestFile::GetName()
{
	_bstr_t name;
	char* pLastSlash = strrchr(_Path, '\\');
	if( pLastSlash == NULL )
		pLastSlash = _Path;
	else
		pLastSlash++;
	name = pLastSlash;
	return name;
}

_bstr_t CTestFile::GetType()
{
	return _Type;
}

_bstr_t CTestFile::GetExt()
{
	char* pExt = strrchr( _Path, '.');
	return pExt;
}

_bstr_t CTestFile::GetCode()
{
	_bstr_t result;
	_Info = "";
	long lineno = 1;
	FILE* f = fopen( _Path, "rt" );
	char bufLine[60002], bufNum[10];
	memset(bufLine,0,60002);
	while( ! feof(f) )
	{
		memset(bufLine, 0, 60002);
		char* pBuf = fgets(bufLine, 60000, f);
		if( pBuf )
		{
			// first make sure we trim the line terminator CR or LF
			char* pEndOfLine = strchr( bufLine, '\r' );
			if( pEndOfLine ) *pEndOfLine = 0;
			pEndOfLine = strchr( bufLine, '\n' );
			if( pEndOfLine ) *pEndOfLine = 0;

			// now store the line in pure and numbered form
			result += bufLine;
			result += "\r\n";

			sprintf(bufNum, "%3d : ", lineno );
			_Info += bufNum;
			_Info += bufLine;
			_Info += "\r\n";
		}
		++lineno;
	}
	fclose( f );

	return result;
}

void CTestFile::AddFixture(CTestFixture* pFixture)
{
	if( pFixture )
	{
		_bstr_t name = pFixture->Name();
		_Fixtures[ name ] = pFixture;
	}
}

CTestFixture* CTestFile::GetFixture(const _bstr_t& name)
{
	FixtureMap::iterator i = _Fixtures.find( name );
	if(  i != _Fixtures.end() )
		return i->second;
	else
		return NULL;
}

void CTestFile::RemoveFixture(CTestFixture* pFixture)
{
	if( pFixture )
	{
		_bstr_t name = pFixture->Name();
		_Fixtures.erase(name);
	}
}


FixtureMap::iterator CTestFile::GetFixturesBegin()
{
	return _Fixtures.begin();
}

FixtureMap::iterator CTestFile::GetFixturesEnd()
{
	return _Fixtures.end();
}


long CTestFile::NumFixtures()
{
	return _Fixtures.size();
}

long CTestFile::NumTests()
{
	return NumFixtures() + 1; // file mainline counts as one fixture
}

long CTestFile::NumFail()
{
	long n = 0;
	for(FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFix = i->second;
		n += pFix->NumFail();
	}
	return n;
}

long CTestFile::NumNotRun()
{
	long n = 0;
	for(FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFix = i->second;
		n += pFix->NumNotRun();
	}
	return n;
}

long CTestFile::NumTicks()
{
	long n = _NumTicks;
	for(FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFix = i->second;
		n += pFix->NumTicks();
	}
	return n;
}



void CTestFile::ResetTrace()
{
	_Trace = "";
}

void CTestFile::AddTrace(const _bstr_t& info)
{
	_Trace += info;
}

_bstr_t CTestFile::GetTrace()
{
	_bstr_t tmp = _Trace;
	tmp += "\r\n";

	for( FixtureMap::iterator i = _Fixtures.begin(); i != _Fixtures.end(); ++i )
	{
		CTestFixture* pFixture = i->second;
		tmp += pFixture->Name();
		tmp += "\r\n";
		tmp += pFixture->GetTrace();
		tmp += "\r\n";
	}
	return tmp;
}

