// ScriptUnit - run unit tests written in scripts
// http://www.xt1.org/ScriptUnit/
//
// Copyright (C) 2005 Christian Mogensen
//
// Released under LGPL - see the file COPYING.TXT for more info.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//  http://www.gnu.org/licenses/lgpl.html

// MainFrm.h : interface of the CMainFrame class
//
/////////////////////////////////////////////////////////////////////////////

#if !defined(AFX_MAINFRM_H__D8A4960B_2E61_11D8_AC46_0050044DE41E__INCLUDED_)
#define AFX_MAINFRM_H__D8A4960B_2E61_11D8_AC46_0050044DE41E__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <atlsplit.h>
#include <atlctrlx.h>		// CPaneContainer
#include "FolderDialogEx.h"
#include "SplashWnd.h"

#include "Observer.h"
#include "TestEngine.h"

#include "StatusView.h"
#include "TestTreeView.h"

class CMainFrame : 
		public CFrameWindowImpl<CMainFrame>, 
		public CUpdateUI<CMainFrame>,
		public CMessageFilter, 
		public CIdleHandler,
		public IObserver
{
public:
	DECLARE_FRAME_WND_CLASS(NULL, IDR_MAINFRAME)

	CSplitterWindow		_vertSplit;
	CHorSplitterWindow	_horSplit;

	CPaneContainer		_TreePane;
	CPaneContainer		_StatusPane;
	CPaneContainer		_TracePane;

	CStatusView			_Status;
	CTestTreeView		_Tree;
	CEdit				_Trace;

	CTestEngine*		_pEngine;

	CMainFrame()
	{
		_pEngine = NULL;
	}

	void SetEngine(CTestEngine* pEngine)
	{
		_pEngine = pEngine;
	}

	virtual BOOL PreTranslateMessage(MSG* pMsg)
	{
		if(CFrameWindowImpl<CMainFrame>::PreTranslateMessage(pMsg))
			return TRUE;

		return FALSE;
		// _TreePane.PreTranslateMessage(pMsg);
	}

	virtual BOOL OnIdle()
	{
		UIUpdateToolBar();
		return FALSE;
	}

	BEGIN_MSG_MAP(CMainFrame)
		MESSAGE_HANDLER(WM_CREATE, OnCreate)
		MESSAGE_HANDLER(WM_DESTROY, OnDestroy)
		MESSAGE_HANDLER(WM_SHOWWINDOW, OnShow)
		MESSAGE_HANDLER(WM_DROPFILES, OnDropFiles)
		COMMAND_ID_HANDLER(ID_APP_EXIT, OnFileExit)
		COMMAND_ID_HANDLER(ID_FILE_OPENDIR, OnFileOpenDir)
		COMMAND_ID_HANDLER(ID_FILE_OPEN, OnFileOpen)
		COMMAND_ID_HANDLER(ID_FILE_SAVE, OnFileSave)
		COMMAND_ID_HANDLER(ID_FILE_REMOVE, OnFileRemove)
		COMMAND_ID_HANDLER(ID_VIEW_TOOLBAR, OnViewToolBar)
		COMMAND_ID_HANDLER(ID_VIEW_STATUS_BAR, OnViewStatusBar)
		COMMAND_ID_HANDLER(ID_APP_ABOUT, OnAppAbout)
		COMMAND_ID_HANDLER(ID_APP_RUNTHIS, OnAppRunThis)
		COMMAND_ID_HANDLER(ID_APP_RUNALL,  OnAppRunAll)
		COMMAND_ID_HANDLER(ID_APP_RUNSTOP, OnAppRunStop)
		COMMAND_ID_HANDLER(ID_APP_EDITTHIS, OnAppEditThis)
		NOTIFY_HANDLER(_Tree.GetDlgCtrlID(), TVN_SELCHANGED, OnSelChanged)
		NOTIFY_HANDLER(_Tree.GetDlgCtrlID(), NM_DBLCLK, OnDblClick)

		CHAIN_MSG_MAP(CUpdateUI<CMainFrame>)
		CHAIN_MSG_MAP(CFrameWindowImpl<CMainFrame>)
	END_MSG_MAP()

	BEGIN_UPDATE_UI_MAP(CMainFrame)
		UPDATE_ELEMENT(ID_VIEW_TOOLBAR, UPDUI_MENUPOPUP)
		UPDATE_ELEMENT(ID_VIEW_STATUS_BAR, UPDUI_MENUPOPUP)
	END_UPDATE_UI_MAP()


	HWND CreatePanes()
	{
		// vertical splitter setup
		// client rect for vertical splitter
		CRect rcVert;
		GetClientRect(&rcVert);
		_vertSplit.Create(m_hWnd, rcVert, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN);
		_vertSplit.m_cxyMin = 125;		// minimum size
		_vertSplit.SetSplitterPos(345);	// from left
		_vertSplit.m_bFullDrag = true;	// ghost bar disabled


		// create the horizontal splitter. Note that vSplit is parent of hzSplit
		CRect rcHor;
		GetClientRect(&rcHor);
		_horSplit.Create(_vertSplit.m_hWnd, rcHor, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN);
		_horSplit.m_cxyMin = 125; // minimum size
		_horSplit.SetSplitterPos(250); // from top
		_horSplit.m_bFullDrag = true; // ghost bar enabled

		// top pane container setup
		_StatusPane.Create(_horSplit.m_hWnd);
		_StatusPane.SetTitle("Status");
		_StatusPane.SetPaneContainerExtendedStyle(PANECNT_NOCLOSEBUTTON);

		// bottom pane container setup
		_TracePane.Create(_horSplit.m_hWnd);
		_TracePane.SetTitle("Trace Log");
		_TracePane.SetPaneContainerExtendedStyle(PANECNT_NOCLOSEBUTTON);

		// left pane container setup
		_TreePane.Create( _vertSplit.m_hWnd);
		_TreePane.SetTitle("Test Files");
		_TreePane.SetPaneContainerExtendedStyle(PANECNT_NOCLOSEBUTTON);


		// top right pane
		_Status.Create( _StatusPane.m_hWnd, rcDefault, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,0, 33 );
		_StatusPane.SetClient( _Status.m_hWnd );

		// bottom right pane
		_Trace.Create( _TracePane.m_hWnd, rcDefault, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN | WS_VSCROLL  | ES_MULTILINE  | ES_AUTOVSCROLL | ES_READONLY );
		_TracePane.SetClient( _Trace.m_hWnd );

		_vertSplit.SetSplitterPane(0, _TreePane);
		_vertSplit.SetSplitterPane(1, _horSplit);
		_horSplit.SetSplitterPane(0, _StatusPane);
		_horSplit.SetSplitterPane(1, _TracePane);

		return _vertSplit.m_hWnd;
	}

	void CreateTheTree()
	{
		_Tree.Create( _TreePane.m_hWnd, rcDefault, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN | TVS_HASBUTTONS | TVS_HASLINES | TVS_LINESATROOT | TVS_SHOWSELALWAYS );
		_TreePane.SetClient( _Tree.m_hWnd );

		for( DirMap::iterator i = _pEngine->GetDirsBegin(); i != _pEngine->GetDirsEnd(); ++i )
		{
			CTestDir* pDir = i->second;
			_Tree.AddDir( pDir );
		}
	}

	
	LRESULT OnCreate(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		CreateSimpleToolBar();
		CreateSimpleStatusBar();

		DragAcceptFiles(TRUE);

		m_hWndClient = CreatePanes();
		CreateTheTree();

		UIAddToolBar(m_hWndToolBar);
		UISetCheck(ID_VIEW_TOOLBAR, 1);
		UISetCheck(ID_VIEW_STATUS_BAR, 1);

		GBroker::Subscribe( this );
		GBroker::Subscribe( &_Status );

		// register object for message filtering and idle updates
		CMessageLoop* pLoop = _Module.GetMessageLoop();
		ATLASSERT(pLoop != NULL);
		pLoop->AddMessageFilter(this);
		pLoop->AddIdleHandler(this);


		return 0;
	}

	LRESULT OnDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		bHandled = FALSE;

		GBroker::Unsubscribe( this );
		GBroker::Unsubscribe( &_Status );

		return 0;
	}


	LRESULT OnSelChanged(int idCtrl, LPNMHDR pnmh, BOOL& bHandled)
	{
		ATLTRACE("OnSelChanged\n");
		LPNMTREEVIEW pnmtv = (LPNMTREEVIEW) pnmh;
		if( pnmtv )
		{
			TVITEM newItem = pnmtv->itemNew;
			HTREEITEM hItem = newItem.hItem;
			CTreeItem item(hItem, &_Tree);
			if( ! item.IsNull() )
			{
				// she's got the look
				UpdateTrace();
				bHandled = TRUE;
			}
			else
			{
				_Trace.SetWindowText( "" );
			}
		}
		return 0;
	}

	LRESULT OnDblClick(int idCtrl, LPNMHDR pnmh, BOOL& bHandled)
	{
		ATLTRACE("OnDblClick\n");
		if( pnmh )
		{
			OnAppRunThis(0, 0, 0, bHandled);
		}
		return 0;
	}


	LRESULT OnShow(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		bHandled = FALSE;
		return 0;
	}

	LRESULT OnDropFiles(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		HDROP hDrop = (HDROP) wParam; 
		char buf[1024];
		int nFiles = DragQueryFile( hDrop, 0xFFFFFFFF, buf, 1024 );
		for(int i = 0; i < nFiles; ++i )
		{
			memset(buf, 0, 1024);
			DragQueryFile( hDrop, i, buf, 1024 );
			_bstr_t filename = buf;
			if( filename.length() > 0 )
			{
				if( _pEngine->IsDir( filename ) )
					AddDir( filename );
				else
					AddFile( filename );
			}
		}

		return 0;
	}

    BOOL UISetState(int nID, DWORD dwState)
	{
		if( nID == 1 )
			_Status.StartTest( dwState );
	}


	void OnNotify(ISubject* pSubject, long id, long val)
	{
	}


	LRESULT OnFileExit(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		PostMessage(WM_CLOSE);
		return 0;
	}

	LRESULT OnFileOpenDir(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CTestDir* pFirstDir = GetSelDir();

		CFolderDialogEx dlg( m_hWnd, _T("Choose a folder to load all the test scripts in it."), BIF_RETURNONLYFSDIRS);
		if( pFirstDir )
			dlg.SetDefaultPath( pFirstDir->GetPath()  );
		
		if( dlg.DoModal() == IDOK ) 
		{
			_bstr_t path = dlg.GetFolderPath();
			CTestDir* pDir = AddDir( path );
			if( ! pDir )
				::MessageBox(m_hWnd, path, "No script files found in folder", 0);
		}

		return 0;
	}

	LRESULT OnFileOpen(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/) 
	{
		CTestDir* pFirstDir = GetSelDir();
		_bstr_t defaultPath;
		if( pFirstDir )
			defaultPath = pFirstDir->GetPath();

		// Passing TRUE as the first ctor arg specifies
		// the File Open dialog instead of the File Save dialog.
		CFileDialog dlg(TRUE, _T("vbs"), defaultPath,  OFN_ALLOWMULTISELECT | OFN_EXPLORER | OFN_HIDEREADONLY,
				 _T("VBScript Files (*.vbs)\0*.vbs\0")
				 _T("JavaScript Files (*.js)\0*.js\0")
				 _T("PerlScript Files (*.pls)\0*.pls\0")
				 _T("All Files (*.*)\0*.*\0"),
				 m_hWnd);


		if( dlg.DoModal() == IDOK ) 
		{
			_bstr_t name = dlg.m_szFileName;
			if( dlg.m_ofn.nFileOffset <= name.length() )
			{
				// single file selected
				CTestFile* pFile = AddFile( name );
				if( ! pFile )
					::MessageBox(m_hWnd, name, "File not loaded", 0);
			}
			else
			{
				// multiple file selected
				CTestDir* pDir = _pEngine->GetDir( name );
				_bstr_t names;

				bool bAdded = true;
				TCHAR* pszName = dlg.m_ofn.lpstrFile + dlg.m_ofn.nFileOffset;
				while( pszName && *pszName )
				{
					_bstr_t filename = pszName;
					if( filename.length() > 0 )
					{
						_bstr_t filepath = name + "\\" + filename;
						CTestFile* pFile = AddFile( filepath );
						if( ! pFile )
						{
							names += pszName;
							names += "\n";
							bAdded = false;
						}
					}
					pszName += filename.length() + 1;
				}
				
				if( ! bAdded )
					::MessageBox(m_hWnd, names, "Some files not loaded", 0);
			}
		}

		return 0;
	}

	LRESULT OnFileSave(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/) 
	{
		// Passing TRUE as the first ctor arg specifies
		// the File Open dialog instead of the File Save dialog.
		CFileDialog dlg(FALSE, _T("ScriptUnit log.xml"), NULL,   OFN_OVERWRITEPROMPT | OFN_EXPLORER | OFN_HIDEREADONLY,
				 _T("XML files  (*.xml)\0*.xml\0")
				 _T("All Files (*.*)\0*.*\0"),
				 m_hWnd);
		if( dlg.DoModal() == IDOK )
		{
			_bstr_t name = dlg.m_szFileName;
			_pEngine->SaveTrace( name );
			::MessageBox(m_hWnd, name, "Saved log to file", 0);
		}
		return 0;
	}

	LRESULT OnFileRemove(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/) 
	{
		CTreeItem selItem = _Tree.GetSelectedItem();
		CTestBase* pItem = (CTestBase*) selItem.GetData();
		if( ! selItem.IsNull() && pItem )
		{
			CTestFile* pFile = dynamic_cast<CTestFile*>( pItem );
			CTestDir*  pDir  = dynamic_cast<CTestDir*> ( pItem );
			if( pFile )
			{
				_pEngine->RemoveFile( pFile );
			}
			else 
			{
				if( pDir )
				{
					_pEngine->RemoveDir( pDir );
				}
			}
		}
		
		_Trace.UpdateWindow();		
		return 0;
	}

	LRESULT OnViewToolBar(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		BOOL bVisible = !::IsWindowVisible(m_hWndToolBar);
		::ShowWindow(m_hWndToolBar, bVisible ? SW_SHOWNOACTIVATE : SW_HIDE);
		UISetCheck(ID_VIEW_TOOLBAR, bVisible);
		UpdateLayout();
		return 0;
	}

	LRESULT OnViewStatusBar(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		BOOL bVisible = !::IsWindowVisible(m_hWndStatusBar);
		::ShowWindow(m_hWndStatusBar, bVisible ? SW_SHOWNOACTIVATE : SW_HIDE);
		UISetCheck(ID_VIEW_STATUS_BAR, bVisible);
		UpdateLayout();
		return 0;
	}

	LRESULT OnAppAbout(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
	{
		CAboutDlg dlg;
		dlg.DoModal();
		return 0;
	}

	LRESULT OnAppRunThis(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
	{
		CTreeItem selItem = _Tree.GetSelectedItem();
		CTestBase* pItem = (CTestBase*) selItem.GetData();
		if( ! selItem.IsNull() && pItem )
		{
			CTestFixture* pFixture	= dynamic_cast<CTestFixture*>( pItem );
			CTestFile* pFile		= dynamic_cast<CTestFile*>( pItem );
			CTestDir*  pDir			= dynamic_cast<CTestDir*> ( pItem );

			if( pFixture)
				_pEngine->RunThisTest( pFixture );
			if( pFile )
				_pEngine->RunThisTest( pFile );
			else if( pDir )
				_pEngine->RunThisTest( pDir );

			UpdateTrace();
		}
		else
			OnAppRunAll(wNotifyCode, wID, hWndCtl, bHandled);
		
		UpdateWindow();
		return 0;
	}

	LRESULT OnAppRunAll(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
	{
		if( _pEngine )
		{
			_pEngine->RunAllTests();
			UpdateTrace();
		}

		return 0;
	}

	LRESULT OnAppRunStop(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
	{
		if( _pEngine )
		{
			_pEngine->StopTest();
			UpdateTrace();
		}

		return 0;
	}

	LRESULT OnAppEditThis(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
	{
		_bstr_t path = "";
		CTreeItem selItem = _Tree.GetSelectedItem();
		CTestBase* pItem = (CTestBase*) selItem.GetData();
		if( ! selItem.IsNull() && pItem )
		{
			CTestDir* pDir		   = dynamic_cast<CTestDir*>( pItem );
			CTestFile* pFile	   = dynamic_cast<CTestFile*>( pItem );
			CTestFixture* pFixture = dynamic_cast<CTestFixture*>( pItem );
			if( pFixture )
				pFile = pFixture->GetFile();
			if( pFile )
			{
				path = pFile->GetPath();
				HINSTANCE hInst = ::ShellExecute( m_hWnd, "edit", path, "", pFile->GetDir()->GetPath(), SW_SHOW );
			}
			if( pDir && ! pFile )
			{
				path = pDir->GetPath();
				HINSTANCE hInst = ::ShellExecute( m_hWnd, "open", path, "", path, SW_SHOW );
			}
		}
		return 0;
	}

private:
	CTestDir* AddDir( const _bstr_t& path )
	{
		CTestDir* pDir = _pEngine->AddDir( path );
		if( pDir )
		{
			_Tree.AddDir( pDir );
			pDir->GetItem().Select();
		}
		return pDir;
	}

	CTestFile* AddFile( const _bstr_t& path )
	{
		CTestFile* pFile = _pEngine->AddFile( path );
		if( pFile )
		{
			_Tree.AddFile( pFile );
			pFile->GetItem().Select();
			// add sub-items to the tree
			_pEngine->UpdateFixtures( pFile );
		}
		return pFile;
	}

	CTestDir* GetSelDir()
	{
		CTreeItem selItem = _Tree.GetSelectedItem();
		CTestBase* pItem = (CTestBase*) selItem.GetData();
		if( ! selItem.IsNull() && pItem )
		{
			CTestFile* pFile = dynamic_cast<CTestFile*>( pItem );
			CTestDir*  pDir  = dynamic_cast<CTestDir*> ( pItem );
			if( pFile )
				return pFile->GetDir();
			else
				return pDir;
		}
		return NULL;
	}

	void UpdateTrace()
	{
		CTreeItem selItem = _Tree.GetSelectedItem();
		CTestBase* pItem = (CTestBase*) selItem.GetData();
		if( ! selItem.IsNull() && pItem )
		{
			_Trace.SetWindowText( pItem->GetTrace() + "\r\n" + pItem->GetInfo() );
		}
	}

};

/////////////////////////////////////////////////////////////////////////////

//{{AFX_INSERT_LOCATION}}
// Microsoft Visual C++ will insert additional declarations immediately before the previous line.

#endif // !defined(AFX_MAINFRM_H__D8A4960B_2E61_11D8_AC46_0050044DE41E__INCLUDED_)
